# authentic2 - versatile identity manager
# Copyright (C) 2010-2019 Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import logging

from django.core.management.base import BaseCommand

from authentic2.saml.models import SPOptionsIdPPolicy


class Command(BaseCommand):
    def add_arguments(self, parser):
        parser.add_argument("-n", "--name", help="Service options policy name")
        parser.add_argument(
            "-E",
            "--enabled",
            action="store_true",
            dest="enabled",
            help="Enable the SP options policy",
        )

        parser.add_argument(
            "-S",
            "--signed-authn-requests",
            action="store_true",
            dest="authn_request_signed",
            help="Require that authentication requests are signed",
        )

        parser.add_argument(
            "-D",
            "--default-nameid-format",
            default="email",
            dest="default_name_id_format",
            help="Default NameIDFormat",
        )

        parser.add_argument(
            "-A",
            "--accepted-nameid-formats",
            dest="accepted_name_id_formats",
            help="Accepted NameID formats",
        )

        parser.add_argument(
            "-N",
            "--no-confirm",
            action="store_true",
            dest="no_confirm",
            help="Disable confirmation",
        )

    __exiting_attributes = None

    def handle(self, *args, **kwargs):
        root_logger = logging.getLogger()
        logger = logging.getLogger(__name__)

        # ensure log messages are displayed only once on terminal
        stream_handlers = [
            x
            for x in root_logger.handlers
            if isinstance(x, logging.StreamHandler)
            if x.stream.isatty()
        ]
        if stream_handlers:
            handler = stream_handlers[0]
        else:
            handler = logging.StreamHandler()
            logger.addHandler(handler)

        # add timestamp to messages
        formatter = logging.Formatter(fmt="%(asctime)s %(message)s", datefmt="%Y-%m-%d %H:%M:%S")
        handler.setFormatter(formatter)

        verbosity = int(kwargs["verbosity"])
        if verbosity == 1:
            logger.setLevel(logging.ERROR)
        elif verbosity == 2:
            logger.setLevel(logging.INFO)
        elif verbosity == 3:
            logger.setLevel(logging.DEBUG)

        name = kwargs.get("name")
        if not name:
            logger.fatal("You must provide a name using -n/--name parameter")
            return

        default_name_id_format = kwargs.get("default_name_id_format", "email")

        accepted_name_id_formats = [default_name_id_format]
        if kwargs.get("accepted_name_id_formats"):
            orig_accepted_name_id_formats = kwargs.get("accepted_name_id_formats")
            accepted_name_id_formats = [
                item.strip() for item in orig_accepted_name_id_formats.split(",")
            ]

        # Confirmation
        if not kwargs["no_confirm"]:
            answer = input("Do you confirm you want to create this SP options policy? [y/N] ")
            if answer.lower().strip() != "y":
                logger.warning("User canceled")
                return

        SPOptionsIdPPolicy.objects.create(
            name=kwargs.get("name", "default-sp-options-policy"),
            enabled=kwargs.get("enabled", False),
            authn_request_signed=kwargs.get("authn_request_signed", False),
            default_name_id_format=default_name_id_format,
            accepted_name_id_format=accepted_name_id_formats,
        )
