#!/usr/bin/python3
"""Script to initialize Authentic2 OIDC IDP support"""

import argparse
import json
import logging
import sys
import os.path

from jwcrypto import jwk

default_output = "/etc/authentic2/settings.d/11supann-oidc.py"
default_key_type = "RSA"
default_key_length = 4096

parser = argparse.ArgumentParser(description=__doc__)

parser.add_argument("-d", "--debug", action="store_true", help="Debug mode")
parser.add_argument("-q", "--quiet", action="store_true", help="Quiet mode")

parser.add_argument(
    "-o",
    "--output",
    help=f"OIDC configuration output file (default: {default_output})",
    default=default_output
)

parser.add_argument(
    "-t",
    "--key-type",
    help=f"OIDC JWK key type (default: {default_key_type})",
    default=default_key_type
)

parser.add_argument(
    "-l",
    "--key-length",
    type=int,
    help=f"OIDC JWK key length (default: {default_key_length})",
    default=default_key_length
)

args = parser.parse_args()

if os.path.exists(args.output):
    parser.error(f"Output file '{args.output}' already exists.")

logging.basicConfig(
    level=logging.DEBUG if args.debug else (logging.WARNING if args.quiet else logging.INFO)
)

logging.info("Initialize Authentic2 OIDC IDP support in %s", args.output)

logging.info("Generate a %s key (length: %d)...", args.key_type, args.key_length)
k = jwk.JWK.generate(kty=args.key_type, size=args.key_length)
logging.info("Done.")

logging.info("Export JWK set in %s...", args.output)
s = jwk.JWKSet()
s['keys'].add(k)

with open(args.output, "w", encoding="utf8") as fd:
    fd.write("# vim: shiftwidth=4 tabstop=4 expandtab\n\n")
    fd.write(f"A2_IDP_OIDC_JWKSET = {json.dumps(s.export(as_dict=True), indent=4)}")
logging.info("Done.")

logging.info(
    "Initialize Authentic2 OIDC IDP support initialized. Restart authentic2 service to apply."
)
