import { Subscriptions } from '@rocket.chat/models';

import { i18n } from '../../../../../server/lib/i18n';
import { isRoomCompatibleWithVideoConfRinging } from '../../../../../server/lib/isRoomCompatibleWithVideoConfRinging';
import { roomCoordinator } from '../../../../../server/lib/rooms/roomCoordinator';
import { settings } from '../../../../settings/server';

const CATEGORY_MESSAGE = 'MESSAGE';
const CATEGORY_MESSAGE_NOREPLY = 'MESSAGE_NOREPLY';

function enableNotificationReplyButton(room, username) {
	// Some users may have permission to send messages even on readonly rooms, but we're ok with false negatives here in exchange of better perfomance
	if (room.ro === true) {
		return false;
	}

	if (!room.muted) {
		return true;
	}

	return !room.muted.includes(username);
}

export async function getPushData({
	room,
	message,
	userId,
	senderUsername,
	senderName,
	notificationMessage,
	receiver,
	shouldOmitMessage = true,
}) {
	const username = settings.get('Push_show_username_room') ? (settings.get('UI_Use_Real_Name') && senderName) || senderUsername : '';

	const lng = receiver.language || settings.get('Language') || 'en';

	let messageText;
	if (shouldOmitMessage && settings.get('Push_request_content_from_server')) {
		messageText = i18n.t('You_have_a_new_message', { lng });
	} else if (!settings.get('Push_show_message')) {
		messageText = i18n.t('You_have_a_new_message', { lng });
	} else {
		messageText = notificationMessage;
	}

	return {
		payload: {
			sender: message.u,
			senderName: username,
			type: room.t,
			name: settings.get('Push_show_username_room') ? room.name : '',
			messageType: message.t,
			tmid: message.tmid,
			...(message.t === 'e2e' && { msg: message.msg, content: message.content }),
		},
		roomName:
			settings.get('Push_show_username_room') && roomCoordinator.getRoomDirectives(room.t).isGroupChat(room)
				? `#${await roomCoordinator.getRoomName(room.t, room, userId)}`
				: '',
		username,
		message: messageText,
		badge: await Subscriptions.getBadgeCount(userId),
		category: enableNotificationReplyButton(room, receiver.username) ? CATEGORY_MESSAGE : CATEGORY_MESSAGE_NOREPLY,
	};
}

export function shouldNotifyMobile({
	disableAllMessageNotifications,
	mobilePushNotifications,
	hasMentionToAll,
	isHighlighted,
	hasMentionToUser,
	hasReplyToThread,
	roomType,
	isThread,
	isVideoConf,
	userPreferences,
	roomUids,
}) {
	if (settings.get('Push_enable') !== true) {
		return false;
	}

	if (disableAllMessageNotifications && mobilePushNotifications == null && !isHighlighted && !hasMentionToUser && !hasReplyToThread) {
		return false;
	}

	if (mobilePushNotifications === 'nothing') {
		return false;
	}

	// If the user is going to receive a ringing push notification, do not send another push for the message generated by that videoconference
	if (
		isVideoConf &&
		settings.get('VideoConf_Mobile_Ringing') &&
		isRoomCompatibleWithVideoConfRinging(roomType, roomUids) &&
		(userPreferences?.enableMobileRinging ?? settings.get(`Accounts_Default_User_Preferences_enableMobileRinging`))
	) {
		return false;
	}

	if (!mobilePushNotifications) {
		if (settings.get('Accounts_Default_User_Preferences_pushNotifications') === 'all' && (!isThread || hasReplyToThread)) {
			return true;
		}
		if (settings.get('Accounts_Default_User_Preferences_pushNotifications') === 'nothing') {
			return false;
		}
	}

	return (
		(roomType === 'd' ||
			(!disableAllMessageNotifications && hasMentionToAll) ||
			isHighlighted ||
			mobilePushNotifications === 'all' ||
			hasMentionToUser) &&
		(isHighlighted || !isThread || hasReplyToThread)
	);
}
