#!/bin/bash
# vim: tabstop=4 shiftwidth=4 softtabstop=4 expandtab

INSTALL_BUILD_DEPS=0
INSTALL_BUILD_DEPS_ONLY=0
PURGE_SOURCE_DIRECTORY=0

function usage() {
    [[ $# -gt 0 ]] && echo -e "$*\n" > /dev/stderr
    cat << EOF
Usage: $0 [-h|I|-O|-P|-x]
    -h|--help                       Show this message
    -I|--install-build-deps         Install build dependencies before building package
    -O|--install-build-deps-only    Only install build dependencies
    -P|--purge-sources-directory    Purge debian package sources directory after building package
    -x|--trace                      Enable bash tracing (set -x)
EOF
    [[ $# -gt 0 ]] && exit 1
}

idx=1
while [[ $idx -le $# ]]; do
    OPT=${!idx}
    case $OPT in
        -h|--help)
            usage
            ;;
        -I|--install-build-deps)
            INSTALL_BUILD_DEPS=1
            ;;
        -O|--install-build-deps-only)
            INSTALL_BUILD_DEPS=1
            INSTALL_BUILD_DEPS_ONLY=1
            ;;
        -P|--purge-sources-directory)
            PURGE_SOURCE_DIRECTORY=1
            ;;
        -x|--trace)
            set -x
            ;;
        *)
            usage "Unknown parameter '$OPT'"
    esac
    (( idx+=1 ))
done

# Enter source directory
cd "$( dirname "$0" )"

DEBIAN_RELEASE=$( lsb_release -r -s | sed 's/\..*$//' )

# Install build dependencies
if [[ $INSTALL_BUILD_DEPS -eq 1 ]]; then
    apt-get update
    apt-get install --no-install-recommends --yes \
        devscripts equivs sed rsync git lsb-release wget ca-certificates libfile-mimeinfo-perl
    mk-build-deps \
      --install \
      --tool='apt-get -o Debug::pkgProblemResolver=yes --no-install-recommends --yes' \
      debian/control
fi

# Install build dependencies based on Debian release
if [[ $INSTALL_BUILD_DEPS -eq 1 ]]; then
    # On Debian Stretch, install GitPython using pip because strect version of python3-git have bugs
    if [[ "$DEBIAN_RELEASE" -le 9 ]]; then
        apt-get install --no-install-recommends --yes python3-pip
        python3 -m pip install 'GitPython>=2.15'
    else
        apt-get install --no-install-recommends --yes python3-git
    fi
fi

# Install GPG key (if provided)
if [[ -n "$GPG_KEY" ]]; then
    [[ $INSTALL_BUILD_DEPS -eq 1 ]] && apt-get install --no-install-recommends --yes gnupg2
    [[ $INSTALL_BUILD_DEPS_ONLY -eq 0 ]] && base64 -d <<< "$GPG_KEY" | gpg --import
fi

# Stop here on install build deps only mode
[[ $INSTALL_BUILD_DEPS_ONLY -eq 1 ]] && exit 0

# Check gitdch is installed
GITDCH=$(which gitdch)
set -e
if [[ -z "$GITDCH" ]]; then
    TMP_GITDCH=$(mktemp -d)
    echo "Temporary install gitdch in $TMP_GITDCH"
    if [[ "$DEBIAN_RELEASE" -le 9 ]]; then
        wget -O "$TMP_GITDCH/gitdch" https://gitea.zionetrix.net/bn8/gitdch/raw/tag/python35/gitdch
    else
        wget -O "$TMP_GITDCH/gitdch" https://gitea.zionetrix.net/bn8/gitdch/raw/master/gitdch
    fi
    chmod +x "$TMP_GITDCH/gitdch"
    GITDCH="$TMP_GITDCH/gitdch"
else
    TMP_GITDCH=""
fi

# Detect package name from debian/control file
PACKAGE_NAME=$( grep -E ^Source: debian/control | sed 's/^Source: //' )

# Clean previous build
rm -fr ./*-build-deps_* ./dist

# Compute version using git describe
# Note : If no tag exist, git describe will fail: in this case, compute a 0.0 version with same
# format as git describe
VERSION="$( git describe --tags 2> /dev/null )" || \
    VERSION="0.0-$( git log --oneline | wc -l )-$( git describe --tags --always )"

# Remove EE branch tags suffix
# shellcheck disable=SC2001
VERSION=$( sed 's/-ee[0-9]\{2,3\}$//' <<< "$VERSION" )

# Fix version format to match with Python specs
# See: https://peps.python.org/pep-0440/
# shellcheck disable=SC2001
VERSION=$( sed -e 's/^[^0-9]*//' -e 's/^\([0-9][^-]*\)-\(.*\)/\1+\2/' <<< "$VERSION" )

# Compute EE debian codename
DEBIAN_CODENAME=$( lsb_release -c -s )
[[ "$DEBIAN_RELEASE" -ge 9 ]] && DEBIAN_CODENAME="${DEBIAN_CODENAME}-ee"

# Compute debian package version by adding EE debian version suffix
DEB_VERSION_SUFFIX="~ee${DEBIAN_RELEASE}0+1"
DEB_VERSION="${VERSION}${DEB_VERSION_SUFFIX}"

# Create dist directory
mkdir dist
DIST_DIR=dist/$PACKAGE_NAME-$DEB_VERSION
rsync -av \
    --exclude 'dist' \
    --exclude '.git*' \
    --exclude 'build.sh' \
    --exclude 'docker-builds.sh' \
    ./ "$DIST_DIR/"

# Set version in package files
mapfile -t FILES < <( awk '{print $1}' debian/*install | sed  's/\*//' | xargs find )
[[ "${#FILES[@]}" -eq 0 ]] && echo "No file found in debian/*install files!" && exit 1
for FILE in "${FILES[@]}"; do
    case "$( mimetype -b "$FILE" )" in
        application/x-perl)
            echo "  - in $FILE (perl)"
            sed -i "s/^my \$version *=.*dev.*$/my \$version = '$VERSION';/" "$DIST_DIR/$FILE"
            ;;
        application/x-shellscript)
            echo "  - in $FILE (shellscript)"
            sed -i "s/^VERSION=dev$/VERSION='$VERSION'/" "$DIST_DIR/$FILE"
            ;;
    esac
done

# Compute gitdch extra args
GITDCH_EXTRA_ARGS=()
[[ -n "$DEBFULLNAME" ]] && GITDCH_EXTRA_ARGS+=( "--maintainer-name" "$DEBFULLNAME" )
[[ -n "$DEBEMAIL" ]] && GITDCH_EXTRA_ARGS+=( "--maintainer-email" "$DEBEMAIL" )

if [[ -e debian/changelog ]]; then
    LAST_MANUAL_COMMIT_ID=$( git log --oneline -n1 -- debian/changelog|awk '{print $1}' )
    GITDCH_EXTRA_ARGS+=( "--append" "--revision" "${LAST_MANUAL_COMMIT_ID}..HEAD" )
fi

# Generate debian changelog using generate_debian_changelog.py
$GITDCH \
    --package-name "$PACKAGE_NAME" \
    --version "${DEB_VERSION}" \
    --version-suffix "${DEB_VERSION_SUFFIX}" \
    --code-name "$DEBIAN_CODENAME" \
    --output "$DIST_DIR/debian/changelog" \
    --release-notes dist/release-notes.md \
    --clean-tags-regex '\-ee[0-9]{2,3}$' \
    --exclude "^CI: " \
    --exclude "\.gitlab-ci(\.yml)?" \
    --exclude "build(\.sh)?" \
    --exclude "README(\.md)?" \
    --exclude "^Merge branch " \
    --verbose "${GITDCH_EXTRA_ARGS[@]}"

# Clean temporary gitdch installation
[[ -n "$TMP_GITDCH" ]] && rm -fr "$TMP_GITDCH"

# Build debian package
BUILD_ARGS=()
if [[ -z "$GPG_KEY" ]]; then
    BUILD_ARGS+=("--no-sign")
else
    echo "GPG key provide, enable package signing."
fi
cd "$DIST_DIR/"
if dpkg-buildpackage "${BUILD_ARGS[@]}"; then
    # Handle PURGE_SOURCE_DIRECTORY option
    [[ $PURGE_SOURCE_DIRECTORY -eq 0 ]] || rm -fr "../../$DIST_DIR"
else
    echo "Failed to build debian package"
    exit 1
fi
