#!/bin/bash
#
# ee-installer-libs
#
# Copyright (C) 2016 "Pascal Brugier" <pbrugier@easter-eggs.com>
#
# Distributed under terms of the GPLv4 license.
#

# Check function args quantity {{{
function check_args () {

  local _SCRIPT="${1}"
  local _FNAME="${2}"
  local _EXPECT_QTT="${3}"
  local _GOT_QTT="${4}"

  [ ${_EXPECT_QTT} -ne ${_GOT_QTT} ] && log fatal "${_SCRIPT}: ${_FNAME} expecting ${_EXPECT_QTT} args got ${_GOT_QTT}"
}
#}}}

# Check var {{{
function check_var () {
  local _VAR_STATUS="${1}"
  local _VAR_NAME="${2}"
  local _VAR_VALUE="${3}"
  local _ERROR="NO"
  local _LOG_MSG

  log info "Checking var ${_VAR_NAME} value"
  case ${_VAR_STATUS} in
    "UNSET")
      _LOG_MSG="Var ${_VAR_NAME} is not set"
      _ERROR="YES"
      ;;
    "EMPTY")
      _LOG_MSG="Var ${_VAR_NAME} is set with empty value"
      _ERROR="YES"
      ;;
  esac

  case ${_ERROR} in
    "YES")
      log fatal "Error: ${_LOG_MSG}. Exiting"
      ;;
    "NO")
      log info "Var ${_VAR_NAME} is set to ${_VAR_VALUE}"
      ;;
  esac
}
#}}}

# Source configs file {{{
function source_conf {

  local _CONF_DIR=${1}
  [ ${#} -eq 2 ] && local _DEBUG=${2}
  # Set _DEBUG default value if none from command line
  _DEBUG=${_DEBUG:-QUIET}

  [ ! -d ${_CONF_DIR} ] && echo "Error dir ${_CONF_DIR} not present, exiting" && exit 0

  declare -a _NOT_SOURCED_DIR=()
  declare -a _NOT_SOURCED_FILE=()
  declare -a _SOURCED_FILE=()

  # We must source config before loging
  for _name in `find ${_CONF_DIR} -mindepth 1 -maxdepth 1 | sort`
  do
    if [ -d $_name ]
    then
      _NOT_SOURCED_DIR+=($_name)
    elif [[ $_name != *.conf ]]
    then
      _NOT_SOURCED_FILE+=($_name)
    else
      _SOURCED_FILE+=($_name)
      source $_name
    fi
  done

  # Then we log if _DEBUG
  if [ "${_DEBUG}" == "DEBUG" ]
  then
    if [ ${#_NOT_SOURCED_DIR[@]} -gt 0 ]
    then
      for _name in ${_NOT_SOURCED_DIR[*]}
      do
        log debug "${FUNCNAME}: Warn, we do'nt source ${_CONF_DIR} sub dirs ($_name)"
      done
    fi
    if [ ${#_NOT_SOURCED_FILE[@]} -gt 0 ]
    then
      for _name in ${_NOT_SOURCED_FILE[*]}
      do
        log debug "${FUNCNAME}: Warn, we do'nt source file not matching *.conf ($_name)"
      done
    fi
    if [ ${#_SOURCED_FILE[@]} -gt 0 ]
    then
      for _name in ${_SOURCED_FILE[*]}
      do
        log debug "sourced $_name"
      done
    fi
  fi

}
#}}}

# Check hardware type [desktop|laptop]{{{
#
# Echo : desktop or laptop
function detect_hardware_type () {

  local _IS_LAPTOP='NO'
  local _COMPUTER_TYPE="desktop"

  if [ -e `which laptop-detect` ]
  then
    laptop-detect
    res=$?
    if [ $res -eq 0 ]
    then
      _IS_LAPTOP='YES'
      _COMPUTER_TYPE="laptop"
    elif [ $res -gt 1 ]
    then
      log error "${FUNCNAME}: error while running laptop-detect. (err: $res)"
    fi
  else
    log fatal "${FUNCNAME}: command laptop-detect is missing."
  fi

  echo ${_COMPUTER_TYPE} ${_IS_LAPTOP}

}
#}}}

# Disable IPV6 while installing {{{
function disable_ipv6 () {

  run_and_log 'echo 1 > /proc/sys/net/ipv6/conf/all/disable_ipv6'
  res=$?
  [ $res -ne 0 ] && log warn "${FUNCNAME}: Error disabling IPV6 on all interfaces (err $res)"

}
#}}}

# Manage journald wall log output {{{
manage_journald_output () {

  local _ACTION=${1}
  local _DEV_JOURNALD_TTY=${2}

  local _DEV_JOURNALD_TTY_NAME=${_DEV_JOURNALD_TTY/\/dev\/}

  case ${_ACTION} in
    disable)

      log info "Stopping journald wall login during install, systemd log only available on tty1"
      run_and_log 'sed -i.back "s/^#\(ForwardToConsole=\)no/\1yes/;
                                s/^#\(ForwardToWall=\)yes/\1no/;
                                s@^#\(TTYPath=/dev/\)console@\1${_DEV_JOURNALD_TTY_NAME}@" /etc/systemd/journald.conf'
      res=$?
      [ $res -ne 0 ] && log warn "${FUNCNAME}: Error while modifying /etc/systemd/journald.conf"
      run_and_log 'systemctl restart systemd-journald.service'
      res=$?
      [ $res -ne 0 ] && log warn "${FUNCNAME}: Error while restarting journald with systemd"

      log info "Stopping service log to STDOUT during install, logs still available with syslog"
      run_and_log 'sed -i.back "s/^#\(LogTarget=\)journal-or-kmsg/\1journal/;
                                s/^#\(DefaultStandardOutput=\)journal/\1null/" /etc/systemd/system.conf'
      res=$?
      [ $res -ne 0 ] && log warn "${FUNCNAME}: Error while modifying /etc/systemd/system.conf"
      run_and_log 'systemctl daemon-reexec'
      res=$?
      [ $res -ne 0 ] && log warn "${FUNCNAME}: Error while reloading /etc/systemd/system.conf"

      ;;
    enable)
      log info "Reseting service log to STDOUT to origin, systemd log available on all tty now"
      run_and_log 'mv -f /etc/systemd/system.conf.back /etc/systemd/system.conf'
      res=$?
      [ $res -ne 0 ] && log warn "${FUNCNAME}: Error while moving /etc/systemd/system.conf.back to /etc/systemd/system.conf"
      run_and_log 'systemctl daemon-reexec'
      res=$?
      [ $res -ne 0 ] && log warn "${FUNCNAME}: Error while reloading /etc/systemd/system.conf"

      log info "Reseting journald wall login config to origin, systemd log available on all tty now"
      run_and_log 'mv -f /etc/systemd/journald.conf.back /etc/systemd/journald.conf'
      res=$?
      [ $res -ne 0 ] && log warn "${FUNCNAME}: Error while moving /etc/systemd/journald.conf.back to /etc/systemd/journald.conf"
      run_and_log 'systemctl restart systemd-journald.service'
      res=$?
      [ $res -ne 0 ] && log warn "${FUNCNAME}: Error while restarting journald with systemd"

      ;;
  esac
}
#}}}

# Manage systemd log level {{{
manage_systemd_loglevel () {

    local _ACTION=${1}
    case ${_ACTION} in
      info)
        log info "revert back systemd log level to info"
        run_and_log 'sed -i.back "s/^\(LogLevel=\)debug/#\1info/" /etc/systemd/system.conf'
        res=$?
        [ $res -ne 0 ] && log warn "${FUNCNAME}: Error while modifying /etc/systemd/system.conf"
        run_and_log 'systemctl daemon-reexec'
        res=$?
        [ $res -ne 0 ] && log warn "${FUNCNAME}: Error while reloading /etc/systemd/system.conf"
        ;;
      debug)
        log info "Setting systemd log level to debug"
        run_and_log 'sed -i.back "s/^#\(LogLevel=\)info/\1debug/" /etc/systemd/system.conf'
        res=$?
        [ $res -ne 0 ] && log warn "${FUNCNAME}: Error while modifying /etc/systemd/system.conf"
        run_and_log 'systemctl daemon-reexec'
        res=$?
        [ $res -ne 0 ] && log warn "${FUNCNAME}: Error while reloading /etc/systemd/system.conf"

        ;;
    esac
}
# }}}

# Check if hostname is valid {{{
#
# $1 : hostname to check
#
# Echo : 0 if hostname or domain or fqdn are valid, 1 if not
# Return : void
function is_valid_name () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "2" "$#"

  local _TYPE=${1}
  local _VALUE="${2}"

  local _RES=0

  case ${_TYPE} in
    HN)
      [ $( echo "${_VALUE}"|LC_ALL=C grep -Ec "^[a-z0-9\-]{3,}$" ) -eq 1 ] && _RES=1
      ;;
    DN)
      [ $( echo "${_VALUE}"|LC_ALL=C grep -Ec "^([a-z0-9\-]{3,}\.){1,}[a-z0-9]{2,}$" ) -eq 1 ] && _RES=1
      ;;
    FQDN)
      [ $( echo "${_VALUE}"|LC_ALL=C grep -Ec "^[a-z0-9\-]{3,}\.([a-z0-9\-]{3,}\.){1,}[a-z0-9]{2,}$" ) -eq 1 ] && _RES=1
      ;;
  esac

  echo ${_RES}
}
#}}}

# Set hostname {{{
#
# $1 : new hostname (optional, if not set, hostname will be asked)
#
# Echo : if new hostname not provide, asking it
# Return : 0 if host as been rename
function rename_host () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "1" "$#"

  local _NEW_FQDN="${1}"

  if [ -z "${_NEW_FQDN}" ]
  then
    ask_and_rename_host ${DOMAIN_NAME}
    return $?
  fi

  log info "Rename host with new FQDN : ${_NEW_FQDN}"
  local _NEW_HOSTNAME=${_NEW_FQDN%%.*}
  local _NEW_DOMAINNAME=${_NEW_FQDN#*.}

  log debug "Rename host with new name : ${_NEW_FQDN} (Host : ${_NEW_HOSTNAME} / Domain : ${_NEW_DOMAINNAME})"

  hostname ${_NEW_HOSTNAME}
  [ $? -ne 0 ] && log error "Error setting system hostname" && return 1
  export HOSTNAME=${_NEW_HOSTNAME}
  export DOMAINNAME=${_NEW_DOMAINNAME}
  echo ${_NEW_FQDN} > /etc/mailname
  [ $? -ne 0 ] && log error "Error writting mailname" && return 1
  echo ${_NEW_HOSTNAME} > /etc/hostname
  [ $? -ne 0 ] && log error "Error writting hostname" && return 1

  perl -pi.saved -e "s/^127.0.1.1\s+.*/127.0.1.1\t${_NEW_FQDN} ${_NEW_HOSTNAME}/" /etc/hosts
  [ $? -ne 0 ] && log error "Error changing hostname in /etc/hosts" && return 1
  if  [ -f /etc/postfix/main.cf ]
  then
    perl -pi.saved -e "s/^myhostname\s+=\s+.*/myhostname = ${_NEW_HOSTNAME}/" /etc/postfix/main.cf
    [ $? -ne 0 ] && log error "Error changing myhostname in /etc/postfix/main.cf" && return 1
    perl -pi.saved -e "s/^mydomain\s+=\s+.*/mydomain = ${_NEW_DOMAINNAME}/" /etc/postfix/main.cf
    [ $? -ne 0 ] && log error "Error changing mydomain in /etc/postfix/main.cf" && return 1
    log info "Restart Postfix after changing hostname and FQDN"
    run_and_log 'systemctl restart postfix.service'
    res=$?
    [ $? -ne 0 ] && log error "Error while restarting Postfix service. (err: $res)"
  fi

  sed -i "s/^\(#\sHostname\s:\s\).*/\1${_NEW_FQDN}/" ${STATEFILE}

  log info "Host renamed"
  return 0

}
#}}}

# Ask new hostname and rename host {{{
#
# Echo : dialog to ask new hostname
# Return : 0 or 1 one error
function ask_and_rename_host() {

  local _DOMAIN_NAME _NEW_HN _NEW_FQDN

  if [ ${1} ]
  then
    _DOMAIN_NAME="${1}"
    [ $(is_valid_name DN "${_DOMAIN_NAME}") -ne 1 ] && log fatal "${FUNCNAME}: Bad domain name, check configuration file"
  else
    _DOMAIN_NAME="${DOMAIN_NAME}"
  fi

  local _TMP=$( mktemp )
  local _TITLE="Define new hostname"
  local _ASK="Please, define hostname, full FQDN or keep unconfigured :"
  local _INIT="`hostname`.${_DOMAIN_NAME}"
  local _ERR="Hostname defined is not valid !"

  local _FIRST=1

  while [ $(is_valid_name HN "${_NEW_HN}") -ne 1 ]
  do
    [ ${_FIRST} -eq 0 ] && do_msg "${_ERR}" 7 80
    [ ${_FIRST} -eq 1 ] && _FIRST=0

    dialog --title "${_TITLE}" --no-cancel --inputbox "${_ASK}" 0 80 "" 2> ${_TMP}
    if [ $? -eq 0 ]
    then
      if [ ! -s ${_TMP} ]
      then
        dialog --aspect 0 --defaultno --title "${_TITLE}" --yesno "\nHostname not defined, ok ?"  8 80
        if [ $? -eq 0 ]
        then
          _NEW_HN="unconfigured-hostname"
          _DOMAIN_NAME="unconfigured-domain"
          log info "hostname not modifyed, keep default : \"${_NEW_HN}.${_DOMAIN_NAME}\""
          return 2
        elif [ $? -eq 1 ]
        then
          _FIRST=1
          continue
        fi
      else
        if [ $(grep -c "[.]" ${_TMP}) -eq 1 ]
        then
          _NEW_HN="$( cut -d "." -f 1 ${_TMP} )"
          _DOMAIN_NAME="$( cut -d "." -f 2- ${_TMP} )"
        else
          _NEW_HN="$( cat ${_TMP} )"
        fi

        run_and_log 'rm -f ${_TMP}'
        res=$?
        [ $res -ne 0 ] && log error "${FUNCNAME}: Error while removing tempfile ${_TMP}/ (err: $res)"
      fi
    fi
  done

  _NEW_FQDN="${_NEW_HN}.${_DOMAIN_NAME}"

  [ $(is_valid_name FQDN "${_NEW_FQDN}") -ne 1 ] && log fatal "${FUNCNAME}: Invalid hostname: bad FQDN provided"
  rename_host "${_NEW_FQDN}"
  return $?
}
#}}}

# Display state file content {{{
function display_state_file_content() {

  if [ -e  ${STATEFILE} ]
  then
    log info "File content :\n\n`cat ${STATEFILE}`\n"
    log info "State file ${STATEFILE} created"
  else
    log error "${STATEFILE} missing"
  fi

}
#}}}

# Get ENV and FROM while install {{{
#
# Echo ENV and FROM in ARRAY
#
function get_install_type () {

  local _INFO=()

  while read _LINE
  do
     [ $( echo "${_LINE}" | grep -Ec "^#\s+(Version\s+type\s+:\s+|Installed\s+from\s+:\s+)") -eq 0 ] && continue
     _INFO+=(${_LINE//*: /})
  done < ${STATEFILE}

  echo "${_INFO[*]}"
}
#}}}

# Create or update state file {{{
#
# $1 : create or update file
#
# Return 0 if file crzeation or update is ok
function manage_statefile () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "3" "$#"

  local _ACTION=${1}
  local _TYPE=${2}
  local _FROM=${3}

  local _DATE=`date "+%Y/%m/%d at %T"`
  local _VERSION=`dpkg -l ${INSTALLER_PKG_NAME} | grep "^ii" | awk '{print $3}'`

  if [ "${_ACTION}" == "CREATE" ]
  then
    echo "## Debian-installer version used to install this box" > ${STATEFILE}
    cat ${DI_VERSION_FILE} >> ${STATEFILE}
    echo ""  >> ${STATEFILE}
    echo "## ${INSTALLER_PKG_NAME} executed on this box" >> ${STATEFILE}
    echo "# Hostname : `hostname`" >> ${STATEFILE}
    echo "# Package version : ${_VERSION}" >> ${STATEFILE}
    echo "# Last package update : ${_DATE}" >> ${STATEFILE}
    echo "# Version type : ${_TYPE}" >> ${STATEFILE}
    echo "# Installed from : ${_FROM}" >> ${STATEFILE}
    [ $? -ne 0 ] && log error "Can't add state file ${STATEFILE}" && exit 1
  elif [ "${_ACTION}" == "UPDATE" ]
  then

    if [ "$( egrep "^#\sPackage\sversion\s:\s*" ${STATEFILE} | awk '{print $5}' )" != "${_VERSION}" ]
    then
      log info "Modifying package version number and date in state file ${STATEFILE}"
      sed -i "s/^\(#\sPackage\sversion\s:\s\).*/\1${_VERSION}/ ; s@^\(#\sLast\spackage\supdate\s:\s\).*@\1${_DATE}@" ${STATEFILE}
      [ $? -ne 0 ] && log error "Can't modify state file ${STATEFILE}" && exit 1
    fi

    _TYPE_DEFINED=`grep -Ec "^#\sVersion\stype\s:\s((pre-)?production|developpement)" ${STATEFILE}`

    if [ ${_TYPE_DEFINED} -eq 1 ]
    then
      sed -i "s@^\(#\sLast\spackage\supdate\s:\s\).*@\1${_DATE}@ ; s/^\(#\sVersion\stype\s:\s\).*/\1${_TYPE}/" ${STATEFILE}
      [ $? -ne 0 ] && log error "Can't modify state file ${STATEFILE}" && exit 1
    elif [ ${_TYPE_DEFINED} -eq 0 ]
    then
      echo "# Version type : ${_TYPE}" >> ${STATEFILE}
      [ $? -ne 0 ] && log error "Can't add version type to state file ${STATEFILE}" && exit 1
    fi

    _FROM_DEFINED=`grep -Ec "^#\sInstalled\sfrom\s:\s(int|ext)ernal" ${STATEFILE}`

    if [ ${_FROM_DEFINED} -eq 1 ]
    then
      sed -i "s@^\(#\sLast\spackage\supdate\s:\s\).*@\1${_DATE}@ ; s/^\(#\sInstalled\sfrom\s:\s\).*/\1${_FROM}/" ${STATEFILE}
      [ $? -ne 0 ] && log error "Can't modify state file ${STATEFILE}" && exit 1
    elif [ ${_FROM_DEFINED} -eq 0 ]
    then
      echo "# Installed from : ${_FROM}" >> ${STATEFILE}
      [ $? -ne 0 ] && log error "Can't add version type to state file ${STATEFILE}" && exit 1
    fi

  fi
}
#}}}

# Check if it's first host boot {{{
#
# $1 : force mode (YES to force)
#
# Echo : 1 if it's first boot on prod, 2 on preprod, 3 on developpement  0 if not
# Return : void
function check_first_boot() {

  if [ -e ${TYPEFILE} ]
  then

     _TYPE=`check_version_type`

    if [ "${_TYPE}" == "2" ]
    then
      log error "No version type info in ${TYPEFILE}"
    elif [ "${_TYPE}" == "production" ]
    then
      _ENV_OPT="-a"
    elif [ "${_TYPE}" == "pre-production" ]
    then
      _ENV_OPT="-P"
    elif [ "${_TYPE}" == "developpement" ]
    then
      _ENV_OPT="-D"
    fi

     _FROM=`check_installed_from`

     if [ "${_FROM}" == "2" ]
     then
       log error "No installed from info in ${TYPEFILE}"
     elif [ "${_FROM}" == "external" ]
     then
       _FROM_OPT="-e"
     elif [ "${_FROM}" == "internal" ]
     then
       _FROM_OPT="-i"
     fi
  else
    log error "${TYPEFILE} is missing" && exit 1
  fi

  if [ ! -e ${STATEFILE} ]
  then
    manage_statefile CREATE ${_TYPE} ${_FROM}
    echo ${_ENV_OPT} ${_FROM_OPT} CREATE
  else
    manage_statefile UPDATE ${_TYPE} ${_FROM}
    echo 0
  fi

}
#}}}

# Check if we work for prod, preprod or developpement {{{
#
# Echo : production, pre-production or 2 if error
# Return : void
function check_version_type () {

if [ $( grep -Ec "^#\sVersion\stype\s:\s((pre-)?production|developpement)" "${TYPEFILE}" ) -ne 1 ]
  then
    echo 2
  else
      _VERSION_TYPE=`grep -E "^#\sVersion\stype\s:\s((pre-)?production|developpement)" ${TYPEFILE} | awk '{print $5}'`
    echo "${_VERSION_TYPE}"
  fi
}
#}}}

# Get the env version type {{{
#
# Echo : prod, preprod, dev or 2 if error
# Return : void
function get_version_type () {

  local _VERSION_TYPE
  local _SHORT_TYPE_NAME

  if [ $( grep -Ec "^#\sVersion\stype\s:\s((pre-)?production|developpement)" "${TYPEFILE}" ) -ne 1 ]
  then
    echo 2
  else
    _VERSION_TYPE=`grep -E "^#\sVersion\stype\s:\s((pre-)?production|developpement)" ${TYPEFILE} | awk '{print $5}'`
  fi

  case  ${_VERSION_TYPE} in
    "production")
      _SHORT_TYPE_NAME="prod"
      ;;
    "pre-production")
      _SHORT_TYPE_NAME="preprod"
      ;;
    "developpement")
      _SHORT_TYPE_NAME="dev"
      ;;
  esac

  echo "${_SHORT_TYPE_NAME}"
}
#}}}

# Check if installed from LAN (internal) or WAN (external) {{{
#
# Echo : internal, external or 2 if error
# Return : void
function check_installed_from () {

if [ $( grep -Ec "^#\sInstalled\sfrom\s:\s(int|ext)ernal" "${TYPEFILE}" ) -ne 1 ]
  then
    echo 2
  else
    _FROM_TYPE=`grep -E "^#\sInstalled\sfrom\s:\s(int|ext)ernal" ${TYPEFILE} | awk '{print $5}'`
    echo "${_FROM_TYPE}"
  fi
}
#}}}

# Check if hostname must be changed {{{
#
# Echo : 1 if yes, 0 if not
# Return : void
function check_hostname() {
  if [ "$( hostname -s )" == "unconfigured-hostname" ]
  then
    echo 1
  else
    echo 0
  fi
}
#}}}

# Ask user to confirm question or the script will be abort script {{{
#
# $1 : the question. The question will be suffix by " (y/N) ? "
#
# Echo : Asking question and error string
# Return : 0 if user confirmed
function confirm_or_abort () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "1" "$#"

  local _QUESTION=${1}

  while true
  do
    echo -n "${_QUESTION} (y/N) ?"
    read _ANSWER

    if [ "${_ANSWER}" == "y" -o "${_ANSWER}" == "Y" ]
    then
      return 0
    elif [ "${_ANSWER}" == "" -o "${_ANSWER}" == "n" -o "${_ANSWER}" == "N" ]
    then
      echo "Abort."
      exit 0
    else
      echo "Bad answer. You must answer by 'y' or 'n'."
    fi
  done
}
#}}}

# Read answers {{{
function read_yes_no () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "1" "$#"

  _ACTION=${1}

  echo -n "Do you want to ${_ACTION} now (Y/n) ? "
  read a
  if [ "$a" == "n" -o "$a" == "N" ]
  then
    _ANSWER="N"
  elif  [ "$a" == "y" -o "$a" == "Y" ]
  then
    _ANSWER="Y"
  else
    echo "bad choice, only Yy or Nn, retry"
    read_yes_no
  fi
}
#}}}

# Check user that running script {{{
#
# $1 : mandatory user id (default : 0)
# $2 : script runnng by current user
#
# Echo : Error string
# Return : void
function check_user_id () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "2" "$#"

  local _MUST_BE="${1}"
  local _USER_IS="${2}"

  local _MUST_BE_ID="`id -u ${1}`"
  local _USER_IS_ID="`id -u ${2}`"

  log debug "Script must be run by id ${_MUST_BE} and is running by ${_USER_IS}"
  if [ "${_USER_IS_ID}" != "${_MUST_BE_ID}" ]
  then
    log fatal "${FUNCNAME}: This script must be run by user ${_MUST_BE} (id: ${_MUST_BE_ID} ! (Current : ${_USER_IS} / id: ${_USER_IS_ID})"
  fi
}
#}}}

# Apt functions {{{
# Disable d-i aprt proxy config {{{
#
# Echo : nothing
# Return : 0 if OK, other also
function disable_d-i_apt_proxy () {

  if [ -e /etc/apt/apt.conf ]
  then
      if [ $(egrep -c "^Acquire::http::Proxy\s+\"http.*\";" /etc/apt/apt.conf ) -eq 1 ]
      then
        log info "Disabling Debian Installer apt proxy"
        run_and_log 'sed -i "s/^\(Acquire::http::Proxy\s\+\"http.*\"\;$\)/#\1/" /etc/apt/apt.conf'
        [ $? -ne 0 ] && log info "Can't remove apt proxy config  from /etc/apt/apt.conf"
        return 0
      fi
  fi

}
#}}}

# Configure APT proxy {{{
#
# Echo : nothing
# Return : 0 if OK, other else
function configure_apt_proxy_host () {

  local _APT_PROXY_HOST _APT_PROXY_PORT

  if [ ! -f ${APTPROXYFILE} -o ! -s ${APTPROXYFILE} ]
  then
      log fatal "${FUNCNAME}: ${APTPROXYFILE} not find, contact your admin sys."
  fi

  if [ -z ${APT_PROXY_HOST} -a -z ${APT_PROXY_PORT}]
  then

      if [ ! -f ${APT_PROXY_HOST_CONF} ]
      then

        read _APT_PROXY_HOST _APT_PROXY_PORT < <(cut -d " " -f 5 ${APTPROXYFILE} | cut -d "/" -f 3 | sed 's/:/ /')

        log info "setting apt proxy host : ${_APT_PROXY_HOST} and port : ${_APT_PROXY_PORT}"
        run_and_log 'echo -e "# APT PROXY\nAPT_PROXY_HOST=\"${_APT_PROXY_HOST}\"\nAPT_PROXY_PORT=\"${_APT_PROXY_PORT}\"\n" > ${APT_PROXY_HOST_CONF}'
        [ $? -ne 0 ] && log fatal "${FUNCNAME}: Can't define APT_PROXY_HOST or APT_PROXY_PORT in ${APT_PROXY_HOST_CONF}. (err: $res)"

        log info "Deleting ${APTPROXYFILE}, only used during install"
        run_and_log 'rm -f ${APTPROXYFILE}'
        [ $? -ne 0 ] && log info "Can't remove ${APTPROXYFILE}"
        return 0
      else
        log info "APT proxy config file (${APT_PROXY_HOST_CONF}) allready present."
      fi
  else
      log warning "APT_PROXY_HOST and , or APT_PROXY_PORT allready defined, should'nt !"
  fi

}
#}}}

# Check APT proxy {{{
#
# Echo : nothing
# Return : 0 if OK, other also
function check_apt_proxy () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "1" "$#"

  local _APT_PROXY_HOST_CONF="${1}"
  local _APT_PROXY_HOST _APT_PROXY_PORT

  read _APT_PROXY_HOST _APT_PROXY_PORT < <(grep -v "#"  ${_APT_PROXY_HOST_CONF} | cut -d "=" -f 2 | sed 's/"//g' | tr '\n' ' ')

  if [ -z "${_APT_PROXY_HOST}" -o -z "${_APT_PROXY_PORT}" ]
  then
    log debug "APT proxy not enabled"
    return 0
  fi

  check_remote_service ${_APT_PROXY_HOST} ${_APT_PROXY_PORT} TCP
  res=$?
  if [ $res -eq 0 ]
  then
    log debug "Apt will use proxy ${_APT_PROXY_HOST} on port ${_APT_PROXY_PORT}"
    enable_apt_proxy
  else
    log debug "APT proxy ${_APT_PROXY_HOST} not reachable, disabling apt proxy config"
    log warning "${FUNCNAME}: APT proxy could be unavailalble due to a network interface without link or network misconfiguration. (err: $res)"
    disable_apt_proxy
    return 0
  fi

}
#}}}

# Enable APT proxy {{{
#
# Echo : using log function
# Return : 0 if OK, 1 also
function enable_apt_proxy() {

  local _APT_PROXY_URL="http://${APT_PROXY_HOST}"
  [ ! -z "${APT_PROXY_PORT}" ] && _APT_PROXY_URL="${_APT_PROXY_URL}:${APT_PROXY_PORT}"

  log debug "Enable APT proxy : ${_APT_PROXY_URL}"
  echo "Acquire::http::Proxy \"${_APT_PROXY_URL}\";" > ${APT_PROXY_CONF_FILE}
  [ $? -ne 0 ] && log error "Failed to write APT configuration file" && return 1

  export http_proxy="${_APT_PROXY_URL}"

  return 0
}
#}}}

# Disable APT proxy {{{
#
# Echo : nothing
# Return : 0 if OK, other also
function disable_apt_proxy() {
  [ -z "${APT_PROXY_CONF_FILE}" -o ! -e "${APT_PROXY_CONF_FILE}" ] && return 0

  log debug "Disable APT proxy in ${APT_PROXY_CONF_FILE}"
  unset http_proxy
  run_and_log 'sed -i "s/^\(Acquire::http::Proxy\s\+\"http.*\"\;$\)/#\1/" ${APT_PROXY_CONF_FILE}'
  [ $? -ne 0 ] && log error "Failed disabling apt proxy in ${APT_PROXY_CONF_FILE}"

  return $?
}
#}}}

# Check if apt gpg key is allready installed
#
# $1 : gpg key id
# Echo : nothing
# Return : 0 if installed , 1 if not
function is_apt_gpg_key_installed () {

  local _GPG_KEY_ID_LIST="${1}"
  local _GPG_APT_KEYRING_GPG="${2}"
  local _GPG_KEY_ID

  local RETURN=0

  log info "Checking if gpg file ${_GPG_APT_KEYRING_GPG} is present and not empty"
  if [ -s ${_GPG_APT_KEYRING_GPG} ]
  then
    log info "  - File ${_GPG_APT_KEYRING_GPG} exists, we can check if key(s) ${_GPG_KEY_ID_LIST} already in"
    for _GPG_KEY_ID in ${_GPG_KEY_ID_LIST}
    do
      log info "Checking if gpg key ${_GPG_KEY_ID} allready in keyring"

      if [ $(gpg -q --list-options show-only-fpr-mbox --import-options show-only --import < ${_GPG_APT_KEYRING_GPG} | grep -c "${_GPG_KEY_ID}") -eq 1 ]
      then
        log info "  - key allready in keyring"
      else
        log info "  - key not installed"
        RETURN=1
      fi
    done
  else
    log info "  - File ${_GPG_APT_KEYRING_GPG} not present, key not installed"
    RETURN=1
  fi

  return ${RETURN}
}
#}}}

# Install APT foreign keyring {{{
#
# Echo : nothing
# Return : void
function install_foreign_keyring() {

  # Check args passed via $@ (array length)
  check_args "${0}" "${FUNCNAME}" "4" "${#@}"

  local _GPG_INFO=("$@")

  local _GPG_TYPE=${_GPG_INFO[0]}
  local _GPG_SRC=${_GPG_INFO[1]}
  local _GPG_APT_KEYRING=${_GPG_INFO[2]}
  local _GPG_COMMENT=${_GPG_INFO[3]}

  local _GPG_KEY_ID _GPG_ASC_KEY_LIST _GPG_HOME_DIR
  local _GPG_APT_KEYRING_ASC="/tmp/${_GPG_APT_KEYRING}.asc"
  local _GPG_APT_KEYRING_GPG="/etc/apt/trusted.gpg.d/${_GPG_APT_KEYRING}.gpg"

  case ${_GPG_TYPE} in
    KEY_ID)
      _GPG_KEY_ID=${_GPG_SRC}
      is_apt_gpg_key_installed ${_GPG_KEY_ID} ${_GPG_APT_KEYRING_GPG}
      res=$?
      if [ $res -eq 1 ]
      then
        log info "Installing gpg key ${_GPG_KEY_ID} in ${_GPG_APT_KEYRING} for ${_GPG_COMMENT}"
        # Create temp dir for gpg keyring manipulation
        _GPG_HOME_DIR=$(mktemp -d)
        # export  GNUPGHOME
        export GNUPGHOME=${_GPG_HOME_DIR}
        # Import key in temp keyring
        run_and_log 'gpg --keyserver keyserver.ubuntu.com --recv-key ${_GPG_KEY_ID}'
        # Export key in _GPG_APT_KEYRING_GPG
        run_and_log 'gpg --output ${_GPG_APT_KEYRING_GPG} --export  ${_GPG_KEY_ID}'
        # remove temp dir
        run_and_log 'rm -rf ${_GPG_HOME_DIR}'
        res_end=$?
      fi
      ;;
    KEY_URL)
      # Download keyfile
      log info "Downloading ${_GPG_COMMENT} apt gpg key"
      run_and_log 'wget -q ${_GPG_SRC} -O /tmp/${_GPG_APT_KEYRING}.asc'
      # Get keyid list from /tmp/${_GPG_APT_KEYRING}.asc
      _GPG_ASC_KEY_LIST=$(gpg -q --list-options show-only-fpr-mbox --import-options show-only --import < ${_GPG_APT_KEYRING_ASC} | awk '{print $1}')

      # Check if keyid is allready in /etc/apt/trusted.gpg.d/${_GPG_APT_KEYRING}.gpg
      for _GPG_KEY_ID in ${_GPG_ASC_KEY_LIST}
      do
        is_apt_gpg_key_installed ${_GPG_KEY_ID} ${_GPG_APT_KEYRING_GPG}
        res=$?
        res_end=0
        if [ $res -eq 1 ]
        then
          log info "Installing gpg key ${_GPG_KEY_ID} in ${_GPG_APT_KEYRING} for ${_GPG_COMMENT}"
          run_and_log 'gpg -q --no-options --no-default-keyring --no-auto-check-trustdb --no-keyring --import-options import-export --import < ${_GPG_APT_KEYRING_ASC} > ${_GPG_APT_KEYRING_GPG}'
          # Remove temp file ${_GPG_APT_KEYRING_ASC}
          res_end=$?
          # If only one key has been added, all are added, so break
          break
         fi
       done
      ;;
  esac

  if [ $res_end -ne 0 ]
  then
    log warning "${FUNCNAME}: Error while adding gpg key : ${_GPG_KEY_ID}. (err: $res_end)"
  else
    log info " - Ok, done"
  fi

  # Cleanup
  run_and_log 'rm -f ${_GPG_APT_KEYRING_ASC}'

}
#}}}

# APT Upgrade {{{
#
# Echo : apt traces
# Return : void
function apt_upgrade() {
  log info "Upgrade softwares ..."
  log debug "Running : apt-get upgrade --allow-unauthenticated -y"
  log info  "All command ouptut and error redirected to ${LOGFILE}"
  run_and_log 'apt-get upgrade --allow-unauthenticated -y'
  res=$?
  [ $res -ne 0 ] && log fatal "${FUNCNAME}: Error while upgrading packages. (err: $res)"
}
#}}}

# APT Dist Upgrade {{{
#
# Echo : apt traces
# Return : void
function apt_distupgrade () {
  log info "Dist Upgrade softwares ..."
  log debug "Running : apt-get dist-upgrade --allow-unauthenticated -y"
  log info  "All command ouptut and error redirected to ${LOGFILE}"
  run_and_log 'apt-get dist-upgrade --allow-unauthenticated -y'
  [ $res -ne 0 ] && log fatal "${FUNCNAME}: Error while dist-upgrading packages. (err: $res)"
}
#}}}
# APT Clean {{{
function apt_clean () {
  log info "Cleaning apt base ..."
  run_and_log 'apt-get clean'
  res=$?
  [ $res -ne 0 ] && log fatal "${FUNCNAME}: Error while cleaning apt base. (err: $res)"
}
#}}}
# APT Autoremove {{{
function apt_autoremove () {
  log info "Apt autoremove unused packages ..."
  run_and_log 'apt-get -y autoremove'
  res=$?
  [ $res -ne 0 ] && log fatal "${FUNCNAME}: Error while autoremoving unused packages. (err: $res)"
}
#}}}
# Configure APT source.list {{{
#
# Echo : nothing
# Return : void
function configure_apt_sources() {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "3" "$#"

  local _INSTALL_ENV=${1}
  local _OS_CODENAME=${2}
  local _CONF_DIR=${3}

  local _DEST_DIR="/etc/apt/sources.list.d/"
  local _MODIFIED="NO"

  echo "## REPLACED BY FILES in ${_DEST_DIR} ##" > /etc/apt/sources.list

  log info "Copying and setting sources.list file for ${_OS_CODENAME} in ${_INSTALL_ENV} env."

  # Default Debian sources.list
  # Testing if /apt/sources.list-debian/debian.list exists before link
  if [ ! -L ${_DEST_DIR}/debian.list ]
  then
    do_symlink $FUNCNAME ${_CONF_DIR}/apt/sources.list-debian/debian.list ${_DEST_DIR}/
    _MODIFIED="YES"
  fi

  # Customers sources.list
  for TARGET in `ls ${_CONF_DIR}/apt/sources.list-${_INSTALL_ENV}/*.list`
  do
    # Testing if /apt/sources.list-debian/*.list exists before link
    if [ ! -L ${_DEST_DIR}/${TARGET##*/} ]
    then
      do_symlink $FUNCNAME  ${TARGET} ${_DEST_DIR}/
      _MODIFIED="YES"
    fi
  done

  [ "${_MODIFIED}" == "NO" ] && log info "All souces.list files already in place, nothing to do"
}

#}}}

# Update APT cache {{{
#
# Echo : apt traces
# Return : void
function apt_update() {
  log info "Updating package version database"
  run_and_log 'apt-get update'
  res=$?
  [ $res -ne 0 ] && log fatal "${FUNCNAME}: Error while updating package version database (err: $res)"
  log info "Package version database succesfully updated"
}
#}}}

# Update package <package> version{{{
#
# $1 : Package name
# Echo : apt traces
# Return : void
function update_pkg_version () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "1" "$#"

  local _PKG_NAME=${1}

  # First check if new package version is available
  while read _line
  do
    [[ $_line =~ Installed ]] && _installed=$(echo $_line | awk '{print $2}')
    [[ $_line =~ Candidate ]] && _candidate=$(echo $_line | awk '{print $2}')
  done  < <(apt-cache policy ${_PKG_NAME} | grep -E "(Candidate|Installed)")

  if [ $_installed != $_candidate ]
  then
    # Force new config file version even if old is missing.
    local _NON_INTERACTIVE="-o Dpkg::Options::=--force-confnew -o Dpkg::Options::=--force-confmiss"
    log info "Install new version of ${_PKG_NAME}"
    run_and_log 'apt-get install ${_NON_INTERACTIVE} --allow-unauthenticated -y ${_PKG_NAME}'
    res=$?
    [ $res -ne 0 ] && log fatal "${FUNCNAME}: Error while upgrading package ${_PKG_NAME} (err: $res)"
    log info "${_PKG_NAME} successfully upgraded"
  else
    log info "${_PKG_NAME} is already the newest version"
  fi

}
#}}}

# Install dpkg foreign arch {{{
#
# $1 : Arch
# Echo : void
# Return : void
function install_foreign_arch () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "1" "$#"

  local _ARCH=${1}
  foreign_arch=`dpkg --print-foreign-architectures`
  # Install i386 arch
  if [ "$foreign_arch" != "${_ARCH}" ]
  then
    log info "Installing multiarch ${_ARCH} : "
    dpkg --add-architecture ${_ARCH}
    res=$?
    [ $res -ne 0 ] && log error "${FUNCNAME}: Error while add ${_ARCH} arch with dpkg (err: $res)"

    log info "Updating for ${_ARCH} arch"
    apt_update
  else
    log info " Multiarch for ${_ARCH} already installed."
  fi
}
#}}}

# Apt action from file or list {{{
#
# $1 : Array
# if ${#array[@]} = 3
# ${array[0]} : Apt Action
# ${array[1]} : Dist version
# ${array[2]} : [ package name | list file ]
# if ${#array[@]} > 3
# ${array[0]} : Apt Action
# ${array[1]} : Dist version
# ${array[2]} -> Array end : packages list
# Echo : void
# Return : viod
function apt_action () {

  local _ARGS_QTT=$#
  local _ARGS=("$@")

  local _OS_CODENAME=${_ARGS[1]}
  local _TO_REMOVE="NO"

  # Get Apt Action
  case ${_ARGS[0]} in
    task_install_yes)
      _APT_ACTION="-y --allow-unauthenticated -o APT::Install-Recommends=true -o APT::Get::AutomaticRemove=true -o APT::Acquire::Retries=3 install"
      _APT_LOG="forcing install yes from ${_OS_CODENAME}"
      ;;
    force_install_yes)
      _APT_ACTION="install -y --allow-unauthenticated"
      _APT_LOG="forcing install yes from ${_OS_CODENAME}"
      ;;
    allow_downgrades_yes)
      _APT_ACTION="install -y --allow-unauthenticated --allow-downgrades"
      _APT_LOG="forcing install yes from ${_OS_CODENAME}"
      ;;
    force_backport_install_yes)
      _APT_ACTION="install -y --allow-unauthenticated -t ${_OS_CODENAME}"
      _APT_LOG="forcing ${_OS_CODENAME} install yes"
      ;;
    force_non_official_install_yes)
      _APT_ACTION="install -y --allow-unauthenticated -t ${_OS_CODENAME}"
      _APT_LOG="forcing ${_OS_CODENAME} install yes"
      ;;
    purge)
      _APT_ACTION="purge"
      _APT_LOG="purging"
      _TO_REMOVE="YES"
      ;;
    force_purge_yes)
      _APT_ACTION="-y purge"
      _APT_LOG="forcing purge yes"
      _TO_REMOVE="YES"
      ;;
  esac

  local _MY_W_MGS="${FUNCNAME}: Error while ${_APT_LOG}"

  # __FIXME__ GEt all array members after the seconf one
  if [ ${_ARGS_QTT} -gt 3 ]
  then
    # Get package list (starting at array element 3 'till the end)
    _PKG="${_ARGS[@]:2}"
    log info "${_APT_LOG^} package(s) ${_PKG}"
    run_and_log 'apt-get ${_APT_ACTION} ${_PKG}'
    res=$?
    [ $res -ne 0 ] && log warn "${_MY_W_MSG} ${_APT_LOG}  (err: $res)"
  elif [ ${_ARGS_QTT} -eq 3 ]
  then
    _PKG="${_ARGS[2]}"
    # _PKG is a file
    if [ -f ${_PKG} ]
    then
      # Discard empty lines and comment
      # Discard packages not present on the system
      _PKG_LIST=`while read line
                 do
                   [[ ${line} =~ ^#.* ]] || [[ ${line} =~ ^$ ]] && continue
                   dpkg -l ${line} > /dev/null  2>&1
                   res=$?
                   [ $res -ne 0 ] && [ "${_TO_REMOVE}" == "YES" ] && continue
                   printf "%s " ${line}
                 done < ${_PKG}`

      [ -z "${_PKG_LIST}" ] && log info "${_PKG_LIST} : empty file, nothing to do" && return 0
      log info "${_APT_LOG^} packages list from file ${_PKG}"
      run_and_log 'apt-get ${_APT_ACTION} ${_PKG_LIST}'
      res=$?
      [ $res -ne 0 ] && log warn "${_MY_W_MSG} (err: $res)"
    # Force dependncies install
    elif [ "${_PKG}" == "force_dependencies_install" ]
    then
      log info "${_APT_LOG^} for missing dependencies"
      run_and_log 'apt-get ${_APT_ACTION}'
      res=$?
      [ $res -ne 0 ] && log warn "${_MY_W_MSG} for missing dependencies. (err: $res)"
    # _PKG is a package name
    else
      log info "${_APT_LOG^} package ${_PKG}"
      run_and_log 'apt-get ${_APT_ACTION} ${_PKG}'
      res=$?
      [ $res -ne 0 ] && log warn "${_MY_W_MSG} ${_PKG}. (err: $res)"
    fi
  elif [ ${_ARGS_QTT} -eq 2 ]
  then
    _PKG="${_ARGS[1]}"
    if [ -f ${_PKG} ]
    then
      while read line
      do
        # Discard empty lines and comment
        # Discard packages not present on the system
        [[ ${line} =~ ^#.* ]] || [[ ${line} =~ ^$ ]] && continue
        dpkg -l ${line} > /dev/null  2>&1
        res=$?
        [ $res -ne 0 ] && [ "${_TO_REMOVE}" == "YES" ] && continue
        _TASK_NAME=${line}
        log info "${_APT_LOG^} from task ${_TASK_NAME}"
        run_and_log 'apt-get ${_APT_ACTION}'
        res=$?
        [ $res -ne 0 ] && log warn "${_MY_W_MSG} ${_PKG}. (err: $res)"
      done
    else
      log info "${_APT_LOG^} package ${_PKG}"
      run_and_log 'apt-get ${_APT_ACTION} ${_PKG}'
      res=$?
      [ $res -ne 0 ] && log warn "${_MY_W_MSG} ${_PKG}. (err: $res)"
    fi
  else
    log info "${_APT_LOG^}: no package given"
  fi

  return $res
}
#}}}

#}}}

# Update running host with installer{{{
#
# $1 : Installer name
# Echo : apt traces
# Return : void
function update_running_host () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "1" "$#"

  local _INSTALLER=${1}

  log info "Update running host with script ${_INSTALLER}"
  run_and_log '${_INSTALLER} -U'
  res=$?
  [ $res -ne 0 ] && log fatal "${FUNCNAME}: Error while running ${_INSTALLER} -U (err: $res)"
  log info "Running host successfully updated"

}
#}}}

# Keep only the two last linux-image version #{{{
#
# Echo : apt traces
# Return : void
function uninstall_old_kernel () {
  local _VERSION="$(uname -r | awk -F '-virtual' '{ print $1}')"
  local _DONT_REMOVE="linux-headers-virtual|linux-image-virtual|linux-headers-${_VERSION}|linux-image-amd64|linux-image-${_VERSION}"
  # protect "+" for bpo version
  _DONT_REMOVE="${_DONT_REMOVE//+/[+]}"
  local _PKG_LIST=`dpkg --list | grep -Ei "(^ii\s+linux-image|linux-headers)" | awk '/ii/{ print $2}' | grep -Ev "${_DONT_REMOVE}" | sort -V | head -n -1`

  if [ -n "${_PKG_LIST}" ]
  then
    log info "Keeping the two latest linux-images version"
    apt_action force_purge_yes ${_PKG_LIST}
    res=$?
    [ $res -ne 0 ] && log fatal "${FUNCNAME}: Error while purging ${_PKG_LIST} (err: $res)"
    log info "Purging old kernel image end without error"
  fi

  log info "Cleaning apt local cache"
  run_and_log 'apt-get clean'
  res=$?
  [ $res -ne 0 ] && log fatal "${FUNCNAME}: Error while cleaning apt cache (err: $res)"
}
#}}}

# Prepare Sshd tu use /etc/ssh/authorized_keys directory {{{
#
# Return: void
#
function prepare_authorized_keys () {

  local SSH_AUTHKEY_DIR="/etc/ssh/authorized_keys"

  log info "Install ssh keys"
  # check if /etc/ssh/authorized_keys/ exist
  log info "Checking if dir $SSH_AUTHKEY_DIR exists:"
  if [ ! -d $SSH_AUTHKEY_DIR ]
  then
    run_and_log 'mkdir $SSH_AUTHKEY_DIR'
    res=$?
    [ $res -ne 0 ] && log warn "$FUNCNAME: Error while creating $SSH_AUTHKEY_DIR dir. (err: $res)"
    log info "created"
  else
    log info "exists"
  fi

  log info "done"

  # Modify /etc/ssh/sshd_config to use /etc/ssh/authorized_keys/%u
  log info "Modify /etc/ssh/sshd_config to use $SSH_AUTHKEY_DIR/%u"
  if [ $( grep -Ec '^#AuthorizedKeysFile\s+.ssh/authorized_keys\s+.ssh/authorized_keys2' /etc/ssh/sshd_config ) -gt 0 ]
  then
    run_and_log 'sed -i "s|^#\(AuthorizedKeysFile\s\+\).ssh/authorized_keys\s\+.ssh/authorized_keys2|\1$SSH_AUTHKEY_DIR/%u|g" /etc/ssh/sshd_config'
    res=$?
    [ $res -ne 0 ] && log warn "$FUNCNAME: Error while modifying /etc/ssh/sshd_config. (err: $res)"
    log info "updated"
  else
    log info "up to date."
  fi

  log info "Reloading ssh configuration"
  systemctl restart ssh.service
  res=$?
  [ $res -ne 0 ] && log warn "$FUNCNAME: Error while reloading ssh configuration. (err: $res)"
  log info "done."
}
#}}}

# Install user packaged ssh pub key {{{
#
# $1: dir type <SSH_DIR|HOME_DIR>
# $2: user name
# $3: link target
#
# Return: void
function install_user_ssh_pubkey () {

  local _DIR_TYPE=${1}
  local _USERNAME=${2}
  local _KEY_PATH=${3}

  # Define _DEST_DIR and _DEST_FILE depending on _DIR_TYPE
  case ${_DIR_TYPE} in
    SSH_DIR)
      _DEST_DIR=/etc/ssh/authorized_keys/
      _DEST_FILE=${_DEST_DIR}/${_USERNAME}
      ;;
    HOME_DIR)
      _DEST_DIR=/home/${_USERNAME}/.ssh
      _DEST_FILE=${_DEST_DIR}/authorized_keys
      ;;
  esac

  # check if ${_USERNAME} exist
  run_and_log 'getent passwd  ${_USERNAME}'
  res=$?
  [ $res -ne 0 ] && log fatal "${FUNCNAME}: User ${_USERNAME} not present in the system, exiting. (err: $res)"

  # Check ${_DEST_DIR} exist
  if [ ! -d ${_DEST_DIR} ]
  then
    log info "${_DEST_DIR} not present, we'll create."
    run_and_log 'mkdir ${_DEST_DIR}'
    res=$?
    [ $res -ne 0 ] && log fatal "${FUNCNAME}: Error while creating directory ${_DEST_DIR}. (err: $res)"
    log info "${_DEST_DIR} created"
  fi

  # Check if ${_KEY_PATH} file exist
  [ ! -f ${_KEY_PATH} ] && log fatal "${FUNCNAME}: user ssh pub key file ${_KEY_PATH} not present, exiting. (err: $res)"

  # Symlink pubkey from package to destination
  [ -e ${_DEST_FILE} -a -L ${_DEST_FILE} ]
  res=$?
  if [ $res -eq 1 ]
  then
    do_symlink ${FUNCNAME} ${_KEY_PATH} ${_DEST_FILE}
    log info "${_USERNAME} ssh pubkey updated"
  else
    log info "${_USERNAME} ssh pubkey up to date !"
  fi
}
# }}}

# Permit ssh Root login {{{
#
#
#
function permit_root_ssh () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "1" "$#"

  local _IP_ADDR="${1}"
  local _SSHD_CONF=/etc/ssh/sshd_config

  log info "Permit root ssh remote login"
  if [ $(grep -Ec "^PermitRootLogin\s+without-password" ${_SSHD_CONF} ) -eq 1 ]
  then
    log info "Allow root ssh"
    run_and_log 'sed -i "s/^\(PermitRootLogin\s\+\)without-password/\1yes/" ${_SSHD_CONF}'
    res=$?
    if [ $res -ne 0 ]
    then
      log error "${FUNCNAME}: Can't allow ssh root login... (err: $res)"
    else
      log info "Reloading new Ssh configuration"
      run_and_log 'systemctl restart ssh'
      res=$?
      if [ $res -ne 0 ]
      then
        log error "${FUNCNAME}: Error while reloading Ssh configuration. (err: $res)"
      else
        log info "You can have a shell during install via ssh using root@${_IP_ADDR}"
      fi
    fi
  fi
}
#}}}

# Check if user exists _FIXME_ {{{
#
# $1 : user
#
# # Echo : 1 if user exists, 0 if not
# Return : void

#}}}

# Add user in sudoers {{{
function add_users_in_sudoers () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "1" "$#"

  local _USERS_LIST="${1}"
  local _ADD_USER=0
  for user in ${_USERS_LIST}
  do
    getent passwd $user  > /dev/null 2>&1
    [ $? -ne 0 ] && log info "Can't add $user in sudoers, user doesn't exist." && continue

    if [ ! -f /etc/sudoers.d/$user ]
    then
      _ADD_USER=1
    elif [ $(grep -Ec "^$user\s+*" /etc/sudoers.d/$user) -eq 0 ]
    then
      _ADD_USER=1
    else
      log info "User $user already in sudoers"
    fi

    if [ ${_ADD_USER} -eq 1 ]
    then
      log info "Enable sudo rights for user $user : "
      run_and_log 'echo "$user ALL=NOPASSWD:ALL" >> /etc/sudoers.d/$user'
      res=$?
      [ $res -ne 0 ] && log warn "${FUNCNAME}: Error while enabling sudo rights for $user. (err: $res)"
      log info "Chmod 0440 /etc/sudoers.d/$user"
      run_and_log 'chmod 0440 /etc/sudoers.d/$user'
      res=$?
      [ $res -ne 0 ] && log warn "${FUNCNAME}: Error while changing permissions on /etc/sudoers.d/eeadmin. (err: $res)"
    fi

  done
}
#}}}

# Dialog functions {{{
# Dialog YesNo {{{
# OK : 0
# NO : 1
function do_ask {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "3" "$#"

  local _MSG="${1}"
  local _LINE="${2}"
  local _COL="${3}"
  dialog --defaultno --yesno "${_MSG}" ${_LINE} ${_COL}

  local _RESP=$?
  return ${_RESP}
}
# }}}

# Dialog msgbox {{{
# OK : 0
function do_msg {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "3" "$#"

  local _MSG="${1}"
  local _LINE="${2}"
  local _COL="${3}"

  dialog --msgbox "${_MSG}" ${_LINE} ${_COL}

  local _RESP=$?
  return ${_RESP}
}
# }}}

# Dialog Ask Stop {{{
function do_stop_action {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "1" "$#"

  local _MSG="${1}"

  do_ask "${_MSG} ?" "5" "70"
  local _RESP=$?

  return ${_RESP}
}
# }}}

# Dialog create user {{{
function do_create_user () {

  local _TITLE="Ajout d'utilisateur"
  local _TEXT="Confirmez-vous ces informations ?"
  local _USER="${1}"
  local _HOME="${2}"
  local _SHELL="${3}"
  local _COMMENT="${4}"
  local _INFO_FILE="${5}"

  local _resp

  dialog --ok-label "Ok" --colors --title "\Zb\Z5${_TITLE}" \
      --form "${_TEXT}" 15 50 0 \
      "Username:" 1 1 "${_USER}"    1 11 15 0 \
      "Home:"     2 1 "${_HOME}"    2 11 25 0 \
      "Shell:"    3 1 "${_SHELL}"   3 11 10 0 \
      "Comment:"  4 1 "${_COMMENT}" 4 11 40 0 2> ${_INFO_FILE}
  local _RESP=$?

  return ${_RESP}
}
#}}}

# Dialog menu box #{{{
function do_menu () {

  local _TITLE="${1}"
  local _TEXT="${2}"
  local _BTITLE="${3}"
  local _STOP_MSG="${4}"
  # Type can be select or list
  local _TYPE="${5}"
  # Output file
  local _CHOICE_FILE="${6}"
  # Column separator
  local _COL_SEP="${7}"
  # Last element is an array
  local _ARRAY=${8}[@]
  local _LIST=("${!_ARRAY}")

  local _ARRAY_QTT=${#_LIST[*]}
  local _ARRAY_END=$((_ARRAY_QTT-1))


  local _DESC _NUM _option _select _pos _INFO

  unset _option

  local _resp=0

  for _pos in `seq 0 ${_ARRAY_END}`
  do
    #_DESC=${_LIST[$_pos]/ : */}
    _DESC=${_LIST[$_pos]}
    _NUM="$_pos"
    _option+=(${_NUM} "${_DESC}")
  done

  case ${_TYPE} in
    select)
      dialog --column-separator "${_COL_SEP}" --colors --backtitle "${_BTITLE}" --title "\Zb\Z5${_TITLE}" --menu "${_TEXT}" 0 0 0 "${_option[@]}" 2> ${_CHOICE_FILE}
      _resp=$?
      ;;
    list)
      dialog --ok-label "Ok" --colors --backtitle "${_BTITLE}" --title "\Zb\Z5${_TITLE}" --no-tags --no-cancel --menu "${_TEXT}" 0 70 0 "${_option[@]}" 2> ${_CHOICE_FILE}

      _resp=$?
      ;;
  esac

  return $_resp
}

#}}}

# Dailog create password / passphrase {{{
# OK : 0
# NO : 1
function do_create_pass () {

  # Check args passed
  #check_args "${0}" "${FUNCNAME}" "4" "$#"

  local _PASS_TYPE="${1}"
  local _USER="${2}"
  local _MSG="${3}"
  local _ACTION="${4}"
  local _TMP_FILE="${5}"

  local _STOP_MSG="Annuler la création du mot de passe"

  if [ "${_PASS_TYPE}" = "LUKS" -a "${_ACTION}" = "CREATE" ]
  then
    do_msg "\nAprès confirmation du mot de passe, la génération de la clef peut prendre 4 à 5 secondes" "10" "50"
    _USER="Chiffrage Luks"
  fi


  if [ "${_ACTION}" = "CREATE" ]
  then
    first_password=""
    while [ -z "$first_password" ]
    do
      dialog --passwordbox "${_MSG}" 7 70  2> ${_TMP_FILE}
      resp=$?
      if [ $resp -ne 0 ]
      then
        do_stop_action "${_STOP_MSG}"
        resp=$?
        if [ $resp -eq 0 ]
        then
          return $resp
        fi
      else

        local input_password="$(cat ${_TMP_FILE})"
        [ -z "$input_password" ] && continue

        if [ $(echo -n "$input_password" | wc -m) -lt 8 ]
        then
          do_msg "Le mot de passe doit comporter au minimum 8 caractères." "5" "70"
          continue
        elif [ `echo  "$input_password" | grep -c [A-Z] ` -eq 0 -o `echo  "$input_password" | grep -c [0-9]` -eq 0 ]
        then
          do_msg "Le mot de passe doit comporter au minimum une majuscule et un chiffre." "5" "80"
          continue
        else
          first_password=$input_password
        fi
      fi
    done

    # Confirm password
    do_create_pass "${_PASS_TYPE}" "${_USER}" "Confirmer le mot de passe de ${_USER,,}" "CONFIRM" "${_TMP_FILE}"

  elif [ "${_ACTION}" = "CONFIRM" ]
  then
    dialog --no-cancel --passwordbox "${_MSG}" 7 70  2> ${_TMP_FILE}
    resp=$?
    if [ $resp -ne 0 ]
    then
      do_stop_action "${_STOP_MSG}"
      resp=$?
      if [ $resp -eq 0 ]
      then
        return $resp
      fi
    else

      local confirm_password="$(cat ${_TMP_FILE})"
      echo "- $first_password - $confirm_password -" >> /tmp/create_pass.log

      if [ "$confirm_password" = "$first_password" ]
      then
        if [ "${_PASS_TYPE}" = "USER" ]
        then
          chpasswd << EOF
${_USER}:$confirm_password
EOF
          return 1
        elif [ "${_PASS_TYPE}" = "LUKS" ]
        then
          echo "$confirm_password" > ${_TMP_FILE}
        else
          do_msg "Mauvais type d'utilisateur < LUKS | USER >." "5" "70"
        fi
      elif [ -z "$confirm_password" -o "$password" != "$confirm_password" ]
      then
        do_msg  "Les mots de passe ne correspondent pas." "5" "70"
        do_stop_action "${_STOP_MSG}"
        resp=$?
        if [ $resp -eq 0 ]
        then
          return $resp
        else
          do_create_pass "${_PASS_TYPE}" "${_USER}" "Définir le mot de passe de ${_USER}" "CREATE" "${_TMP_FILE}"
        fi
      fi
    fi
  fi
}
# }}}

# Dialog password box (for auth) {{{
function do_auth () {

  local _TITLE="${1}"
  local _TEXT="${2}"
  local _PWD_FILE="${3}"

  local _resp

  dialog --title "${_TITLE}" --clear --passwordbox "${_TEXT}" 10 30 2> ${_PWD_FILE}
  _resp=$?

  return $_resp
}
#}}}

# }}}

# Manage crypted part {{{
# List key slots in use #{{{
function list_keyslot_inuse () {

  local _CRYPTO_DEV="${1}"

  # Get all active key slot
  local _ACTIVE_KEYSLOT=(`cryptsetup luksDump ${_CRYPTO_DEV}  | grep ENABLED | awk '{print $3}' | sed 's/://g'`)
  # Sort hte array by key slot num
  readarray -t _ACTIVE_KEYSLOT_BY_NUM < <(for a in "${_ACTIVE_KEYSLOT[@]}"; do echo "$a"; done | sort -n)

  local _ARRAY_QTT=${#_ACTIVE_KEYSLOT_BY_NUM[*]}
  local _ARRAY_END=$((_ARRAY_QTT-1))

  local  _option _pos _INFO _ACTIVE_KEYSLOT_LIST

  unset _option

  local _resp=0

  for _pos in `seq 0 ${_ARRAY_END}`
  do
    [ $_pos -eq 0 ] && _INFO="Master key, don't remove" || _INFO="Slave key"
    #Define array with results
    _ACTIVE_KEYSLOT_LIST[$_pos]="KeySlot $_pos : ${_INFO}"

  done

  # output array as string using 'declare's representation (minus header
  declare -p _ACTIVE_KEYSLOT_LIST | sed -e 's/^declare -a [^=]*=//'

}
#}}}

# Cryptsetup actions {{{
function cryptsetup_action () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "2" "$#"

  local _ACTION=${1}
  local _DUMP_FILE=${2}

  local _CRYPTO_DEV _LUKS_DEV _DM_CMD _CHOICE_FILE _CHOICE

  local _LUKS_FILE=$( mktemp )

  _CRYPTO_DEV=`lsblk --fs --path --raw | grep -E crypto_LUKS | awk '{print $1}'`
  _LUKS_DEV=`lsblk --fs --path --raw -o NAME,TYPE ${_CRYPTO_DEV} | grep -E "\s?crypt\s?" | awk '{print $1}'`

  _DM_CMD="dmsetup table --target crypt --showkey"

  case ${_ACTION} in
    dump_header)
      run_and_log 'cryptsetup luksHeaderBackup ${_CRYPTO_DEV} --header-backup-file ${_DUMP_FILE}'
      res=$?
      [ $res -ne 0 ] && log warn "${FUNCMANE}: Errror while backuping Luks headers for ${_CRYPTO_DEV}. (err: $res)"
      ;;
    dump_master_key)
      run_and_log '${_DM_CMD} ${_LUKS_DEV} | cut -d " " -f 5 | xxd -r -p > ${_DUMP_FILE}'
      res=$?
      [ $res -ne 0 ] && log warn "${FUNCMANE}: Error while dumping Luks master key for ${_LUKS_DEV}. (err: $res)"
      ;;
    add_secondary_key)
      do_create_pass "LUKS" "Chiffrage LUKS" "Definir le mot de passe pour la seconde clef de chiffrage LUKS\n" "CREATE" "${_LUKS_FILE}"
      run_and_log 'cat ${_LUKS_FILE} | cryptsetup luksAddKey --master-key-file=${_DUMP_FILE} ${_CRYPTO_DEV}'
      res=$?
      [ $res -ne 0 ] && log warn "${FUNCMANE}: Error while adding new Luks passphrase. (err: $res)"
      run_and_log 'rm -f ${_LUKS_FILE}'
      res=$?
      [ $res -ne 0 ] && log warn "${FUNCMANE}: Error while removing ${_INPUT_FILE}. (err: $res)"
      ;;
    list_secondary_keys)
      # ARGS needed by list_keyslot_inuse
      # but no selection needed for list_secondary_keys
      _CHOICE_FILE="NONE"

      # Construct array from array returned by list_keyslot_inuse
      _ACTIVE_KEYSLOT_STRING="$(list_keyslot_inuse ${_CRYPTO_DEV})"

      eval "declare -a _ACTIVE_KEYSLOT=${_ACTIVE_KEYSLOT_STRING}"

      local _MENU_TITLE="LUKS crypt Key(s)"
      local _MENU_TEXT="List of key(s) in use"
      local _STOP_MSG="l'affichage de la liste des clefs"

      # Don't need column separator, send empty one.
      do_menu "${_MENU_TITLE}" "${_MENU_TEXT}" "${_MENU_TITLE}" "${_STOP_MSG}" "list" "${_CHOICE_FILE}" "" _ACTIVE_KEYSLOT

      echo "${_ACTIVE_KEYSLOT_[@]}"
      ;;
    remove_secondary_key)
      do_msg "\nLa suppression de la clef peut prendre 4 à 5 secondes" "8" "50"
      # Pass tmp file to list_keyslot_inuse to get select answer
      _CHOICE_FILE=$( mktemp )

      # Construct array from array returned by list_keyslot_inuse
      _ACTIVE_KEYSLOT_STRING="$(list_keyslot_inuse ${_CRYPTO_DEV})"
      eval "declare -a _ACTIVE_KEYSLOT=${_ACTIVE_KEYSLOT_STRING}"

      local _MENU_TITLE="LUKS crypt Key(s)"
      local _MENU_TEXT="Select a key to remove in the list bellow"
      local _STOP_MSG="la suppression de la clef secodaire"

      # Don't need column separator, send empty one.
      do_menu "${_MENU_TITLE}" "${_MENU_TEXT}" "${_MENU_TITLE}" "${_STOP_MSG}" "select" "${_CHOICE_FILE}" "" _ACTIVE_KEYSLOT
      _CHOICE=$(cat ${_CHOICE_FILE})
      run_and_log 'rm -f ${_CHOICE_FILE})'
      do_auth "Passphrase" "Give Master key passphrase" "${_LUKS_FILE}"
      run_and_log 'cryptsetup --key-file ${_LUKS_FILE} luksKillSlot ${_CRYPTO_DEV} ${_CHOICE}'
      res=$?
      [ $res -ne 0 ] && log warn "${FUNCMANE}: Error while removing LUKS slave key in Slot ${_CHOICE}"
      run_and_log 'rm -f ${_LUKS_FILE}'
      res=$?
      [ $res -ne 0 ] && log warn "${FUNCMANE}: Error while removing tempfile ${_LUKS_FILE}. (err: $res)"
    ;;
  esac

  run_and_log 'rm -f ${_CHOICE_FILE}'
  res=$?
  [ $res -ne 0 ] && log warn "${FUNCMANE}: Error while removing tempfile ${_CHOICE_FILE}. (err: $res)"
}
#}}}
#}}}

# Log management {{{

# Prompt colors {{{
_CRED='\e[1;91m'
_CPURPLE='\e[0;35m'
_CYELLOW='\e[0;33m'
_CGREEN='\e[1;32m'
_CRESET='\e[0m'

function color_log () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "1" "$#"

  level=${1}

  case "$level" in
    error|fatal)
      _COLOR="${_CRED}"
      _ASK_TO_CONTINUE=1
      ;;
    warning)
      _COLOR="${_CYELLOW}"
      _WAIT=4
      ;;
    debug)
      _COLOR="${_CPURPLE}"
      ;;
    info)
      _COLOR="${_CGREEN}"
      ;;
  esac

}
#}}}

# Logs {{{
function log () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "2" "$#"

  local _DATE=`date "+%b  %d %T"`

  _COLOR=""
  _WAIT=0
  _ASK_TO_CONTINUE=0

  if [ $# -gt 1 ]
  then
    _LEVEL="${1}"
    _WORDS="${2}"
    color_log ${_LEVEL}
  elif [ $# -eq 1 ]
  then
    _LEVEL="${1}"
    color_log ${_LEVEL}
    read _WORDS
  else
    read _LEVEL _WORDS
    color_log ${_LEVEL}
  fi

  if [ -n "${_WORDS}" ]
  then
    echo "${_DATE} : [${_LEVEL}] ${_WORDS}" >> ${LOGFILE} 2>&1
    [ "${LOG_TTY}" == "1" ] &&  echo "${_DATE} : [${_LEVEL}] ${_WORDS}" > ${DEV_LOG_TTY} 2>&1
    [ "${_LEVEL}" == "debug" -a "${DEBUG}" != "1" ] && return 0
    # No stdout if QUIET_STDOUT not set or set to 0
    [[ -z ${QUIET_STDOUT+x} || ${QUIET_STDOUT} -eq 0 ]] && echo -e "\n${_COLOR}[${_LEVEL}]${_CRESET} ${_WORDS}"
    [ ${_WAIT} -gt 0 ] && sleep ${_WAIT}
    [ ${_ASK_TO_CONTINUE} -eq 1 ] && echo -n "[Press ENTER to continue else Ctrl C]" && read
  fi

  [ "${_LEVEL}" == "fatal" ] && exit 1
}
#}}}

# Reset LOG_TTY {{{
function manage_log_tty {

  local _ACTION="${1}"
  local _DEV_LOG_TTY="${2}"

  local _DEV_LOG_TTY_NAME=${_DEV_LOG_TTY/\/dev\/}
  local _DEV_LOG_TTY_NUM=${_DEV_LOG_TTY//\/dev\/tty}

  case ${_ACTION} in
    open)
      chvt ${_DEV_LOG_TTY_NUM}
      ;;
    close)
      chvt 1
      systemctl restart getty@${_DEV_LOG_TTY_NAME}.service
      ;;
  esac
}
#}}}

# Exec command and logs its output {{{
#
# Function must be call with "'" (single quote)
# arround the full command
function run_and_log () {
  local _CMD=("$@")
  local _RESULT _RETURN_CODE
  # Log full cmd line with values not vars
  log debug "Run : $(eval "echo '${_CMD[@]}'")"
  # Exec coomand line
  _RESULT=$( eval "${_CMD[@]}" 2>&1 )
  _RETURN_CODE=$?
  log debug "${_RESULT}"
  log debug "Return code : ${_RETURN_CODE}"
  return ${_RETURN_CODE}
}
#}}}

# Exec command as user and logs its output {{{
function runas_and_log () {
  local _USER=${1}
  shift
  local _CMD="${@}"
  local _RESULT _RETURN_CODE
  if [ "${_USER}" == "postgres" ]
  then
    log debug "Run as ${_USER} : ${_CMD//WITH PASSWORD \'*\'\"/WITH PASSWORD '_HIDDEN_'\"}"
  else
    log debug "Run as ${_USER} : $_CMD"
  fi
  _RESULT=$( su - ${_USER} -c "${_CMD}" 2>&1 )
  _RETURN_CODE=$?
  log debug "${_RESULT}"
  log debug "Return code : ${_RETURN_CODE}"
  return ${_RETURN_CODE}
}
#}}}

# Log running script name {{{
function start_log () {

  check_args "${0}" "${FUNCNAME}" "1" "$#"

  local _SCRIPT_NAME="${1}"
  log info "--------------------------------------------------------------------------------------"
  log info "Starting script ${_SCRIPT_NAME}"
}
#}}}
#}}}

# Managed debconf mod <noninteractive|Dialog> {{{
function debconf_frontend () {

  local _DCONF_ACTION=${1}

  if [ "${_DCONF_ACTION}" == "check" ]
  then
    _DCONF_STATE=`grep -C2 debconf/frontend /var/cache/debconf/config.dat | grep Value | awk '{print $2}'`

    [ "${_DCONF_STATE}" != "noninteractive" ] && log fatal "${FUNCNAME}: debconf fronted must be set to noninteractive before preseeding packages"

  elif [ "${_DCONF_ACTION}" == "set" ]
  then


    local _DCONF_MODE=${2}
    local _PRESEED_DIR="${3}"

    [ "${_DCONF_MOD}" == "Dialog" ] && _MSG="back"
    log info "Setting debconf mode ${_MSG} to : ${_DCONF_MODE}"
    run_and_log 'debconf-set-selections < ${_PRESEED_DIR}/frontend-${_DCONF_MODE}.debconf'
  else
    log fatal "${FUNCNAME}: missing args < check | set < noninteractive|Dialog > >"
  fi
}
#}}}

# Load preseed file {{{
#
# $1 : preseed file
#
# Echo : error string
# Return : 0 if preseed is loaded, 1 also
function load_preseed_file () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "1" "$#"

  local _PRESEED_DIR="${1}"

  log info "Preseeding packages"

    debconf_frontend check
  [ ! -d "${_PRESEED_DIR}" ] && log warning "Preseed file ${_PRESEED_DIR} not found" && return 1

  for _PRESEED_FILE_PATH in `find ${_PRESEED_DIR} -type f -name "*.preseed"`
  do

    local _FILE_NAME=${_PRESEED_FILE_PATH##*/}
    local _PKG_NAME=${_FILE_NAME%%.preseed}

    log info "Preseeding package ${_PKG_NAME}."
    run_and_log 'debconf-set-selections < ${_PRESEED_FILE_PATH}'
    [ $? -ne 0 ] && log warning "Error while loading preseed file ${_PRESEED_FILE_PATH}"
  done

  return 0
}
#}}}

# Check if debian package is installed {{{
#
# $1 : package name
#
# Echo : 1 if installed, 0 if not
# Return : void
function check_install_package () {

  # Check args passed
  #check_args "${0}" "${FUNCNAME}" "1" "$#"

  local _PKG=${1}

  local _CMD=`dpkg -l "${PKG}" 2>/dev/null | grep -Ec "^ii\s+${_PKG} "`

  [ "${2}" != "" -a "${2}" == "BPO" ] && _CMD=`dpkg -l "${_PKG}" 2> /dev/null | grep -E "^ii\s+${_PKG} " | grep -Ec "bpo"`

  if [  ${_CMD}  -ne 1 ]
  then
    echo 0
  else
    echo 1
  fi
}
#}}}

# Check if TCP or UDP port is listening {{{
#
# $1 : service name
# $2 : port number
# $3 : timeout
#
# Return : 0 if listening else 1
function wait_port () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "3" "$#"

  local _SERVICE_NAME="${1}"
  local _PORT_NUMBER="${2}"
  local _TIMEOUT="${3}"

  log_daemon_msg "Waiting ${_SERVICE_NAME} listen on port ${_PORT_NUMBER}"

  c=0
  while [ $( netstat -lnp | egrep -c ":+${_PORT_NUMBER}\s+" ) -eq 0 ]
  do
    log_progress_msg "."
    sleep 1
    let c=c+1
    if [ $c -gt ${_TIMEOUT} ]
    then
      log_progress_msg "timeout."
      log_end_msg 1
      return 1
    fi
  done

  log_progress_msg " done"
  log_end_msg 0

  return 0
}
#}}}

# Send mail {{{
#
# $1 : Subject, calling script name : "`basename $0`"
# $2 : Mail body, array : BODY[@]
#
# Return: 0 if ok else 1 via exit_status fatal ${FUNCNAME} 1

function send_mail () {

  local _SUBJECT="${1}"
  local _BODY=("${!2}")

  # Mail Sender address, sender fullname, recipient and default subject
  # part take from /etc/defaul/customer_default_file
  log info "Sending mail"
  sendmail -i -f "${Mail_Sender}" -F "${Sender_Fullname}" "${Mail_To}" <<EOF
to:${Mail_To}
subject: ${Default_Subject} "${_SUBJECT}"
MIME-Version: 1.0
Content-Type: text/plain

Hello,

`date`

${_BODY}

EOF

  res=$?
  if [ $res -ne 0 ]
  then
    log fatal "${FUNCNAME}: Error while sending mail  (err: $res)"
  else
    log info "Mail send (exit status: $res)"
  fi

  return 0
}
#}}}

# Symlink and log {{{
#
# $1: Calling function name
# $2: Target
# $3: Link
function do_symlink () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "3" "$#"


  local _CALL_FROM=${1}
  local _TARGET="${2}"
  local _LINK="${3}"

  run_and_log 'ln -s ${_TARGET} ${_LINK}'
  res=$?
  #[ $res -ne 0 ] && log_and_mail_error "${FUNCNAME} call from ${_CALL_FROM}: Fail to link ${_LINK} to ${_TARGET} (err: $res)"
  [ $res -ne 0 ] && log warn "${FUNCNAME}: called from ${_CALL_FROM} failed to link ${_LINK} to ${_TARGET} (err: $res)"

}
#}}}

# Owner and permissions {{{
# set owner and permissions {{{
#
# $1: Full path to dir or file
# $2: user:group user and group are optionals, : is mandatory
# $3: permissions digit XXX or XXXX
function set_owner_perms () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "3" "$#"

  local _TARGET="${1}"
  local _USER_GROUP="${2}"
  local _PERMS="${3}"

  set_owner "${_TARGET}" "${_USER_GROUP}"
  set_perms "${_TARGET}" "${_PERMS}"

}
#}}}

# set owner {{{
#
# $1: Full path to dir or file
# $2: user:group user and group are optionals, : is mandatory
function set_owner () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "2" "$#"

  local _TARGET="${1}"
  local _USER_GROUP="${2}"

  # ${USER_GROUP} must contain a ":"
#  [[ ! ${_USER_GROUP} =~ ^[a-zA-Z0-9]?:[a-zA-Z0-9]? ]] && log error "${FUNCNAME}: user and groups must be separated by a :"
[ $(echo ${_USER_GROUP} | grep -Ec "^([[:alnum:]]{1,})?:?(:[[:alnum:]]{1,})?$") -ne 1 ] && log error "${FUNCNAME}: user and groups must be separated by a :"

  log info "Seting ${_USER_GROUP} as owner on ${_TARGET}"
  chown ${_USER_GROUP} ${_TARGET} > /dev/null 2>&1
  res=$?
  [ $res -ne 0 ] && log error "${FUNCNAME}: Fail to set ${_USER_GROUP} as owner(s) on ${_TARGET} (err: $res)"
}
#}}}

# set permissions {{{
#
# $1: Full path to dir or file
# $2: permissions digit XXX or XXXX
function set_perms () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "2" "$#"

  local _TARGET="${1}"
  local _PERMS="${2}"

  # ${PERMS} must but digit only
  [ $(echo ${_PERMS} | grep -Ec "^[0-9]{3,4}$") -ne 1 ] && log error "${FUNCNAME}: permissions must be 3 or 4 digits"

  log info "Seting ${_PERMS} permission on ${_TARGET}"
  run_and_log 'chmod ${_PERMS} ${_TARGET}'
  res=$?
  [ $res -ne 0 ] && log error "${FUNCNAME}: Fail to set ${_PERMS} permission on ${_TARGET} (err: $res)"
}
#}}}
#}}}

# Wait for command output {{{
#
# $1 : command, string (could have options but protected with double quote)
# $2 : pattern, string (colud be many words but protected with double quote)
# $3 : grep count output, integer
# $4 : integer comparaison [LT|LE|EQ|GE|GT|NE]
# $5 : timeout, integer
#
# Return : 0 if ok else 1
function wait_command () {

  local _COMMAND="${1}"
  local _PATTERN="${2}"
  local _COUNT=${3}
  local _COMPARE=${4}
  local _TIMEOUT=${5}

  log_daemon_msg "Testing command \"${_COMMAND}\""

  c=0
  while [ $( ${_COMMAND} | egrep -c "${_PATTERN}" ) -${_COMPARE,,} ${_COUNT} ]
  do
    log_progress_msg "."
    sleep 1
    let c=c+1
    if [ $c -gt ${_TIMEOUT} ]
    then
      log_progress_msg "timeout."
      log_end_msg 1
      return 1
    fi
  done

  log_progress_msg " done"
  log_end_msg 0
  _WAIT_CMD_RESULT=$( ${_COMMAND} | egrep "${_PATTERN}" )
  return 0
}
#}}}

# Network functions {{{
# Get connected network interfaces {{{
#
# $1 : TIMEOUT in sec
#
# Echo : iface name
# Return : 0 if ok else 1 via exit_status
function check_iface_link () {

  # unset _IFACE_LIST
  declare -a _IFACE_LIST=()
  local _TIMEOUT=${1}

  # Wait until one or more iface is conected and UP
  wait_command "netstat -i | awk 'NR > 2 { print }'" "BMRU" 1 LT ${_TIMEOUT:-5}
  res=$?
  if [ $res -eq 0 ]
  then
    log info "Network connection is available"
  else
    log fatal "${FUNCNAME}: Interfaces connected : none. (err: $res)"
  fi

  # netstat FLAG :
  # B : A broadcast address has been set
  # M : Supports multicast
  # R : Interface is running (Link detected)
  # U : Interface is up.
  #
  # Awk part remove output header line

  for _iface in `netstat -i | awk 'NR > 2 { print }' | grep 'BMRU' | cut -d " " -f 1`
  do
    _IFACE_LIST+=($_iface)
  done

  if [ ${#_IFACE_LIST[@]} -ge 1 ]
  then
    log info "Network link(s) available on ${_IFACE_LIST[*]}"
    return 0
  else
    log fatal "${FUNCNAME}: Interfaces connected : none. (err: $res)"
  fi
}
#}}}

# Get network info {{{
#
# $1 : default_route|iface_name|iface_addr
# Echo : default route | iface name | iface addr
# Return: 0 if ok else 1 via exit_status
function get_network_info () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "2" "$#"

  # Use timeout option or default to 5 sec.
  local _TIMEOUT=${2}

  wait_command "ip ro sh" "default via" 1 LT ${_TIMEOUT:-5}
  res=$?
  if [ $res -eq 0 ]
  then

    [ -z ${_SEEN} ] && log debug "${FUNCNAME}: ${_WAIT_CMD_RESULT}, TIMEOUT set to ${_TIMEOUT:-5} sec."

    read _DEFAULT_GW _IFACE_DEV < <(echo ${_WAIT_CMD_RESULT} | awk '{print $3, $5}')

    case ${1} in
      default_route)
        if [ -z ${_DEFAULT_GW} ]
        then
          log fatal "${FUNCNAME}: Default route not found. (err: $res)"
        else
          _SEEN="YES"
          for _default_route in $(ip ro sh | grep "default via" | awk '{print $3"-("$5")"}')
          do
            _DEFAULT_ROUTE_LIST+=($_default_route)
          done
          log info "Default route(s) available on ${_DEFAULT_ROUTE_LIST[*]}"
          return 0
        fi
        ;;
      iface_name)
        if [ -z ${_IFACE_DEV} ]
        then
          log fatal "${FUNCNAME}: No device found. (err: $res)"
        else
          _SEEN="YES"
          log info "${FUNCNAME}: There is at least on device available on ${_IFACE_DEV}"
          return 0
        fi
        ;;
      iface_addr)
        _IP_ADDR=`ip ad sh ${_IFACE_DEV} | egrep "^\s+inet\s" | cut -d " " -f 6 | cut -d "/" -f 1`
        if [ -z ${_IP_ADDR} ]
        then
          log fatal "${FUNCNAME}: ${_IFACE_LIST[0]} has no ip address. (err: $res)"
        else
          _SEEN="YES"
          log info "${FUNCNAME}: IP addr : ${_IP_ADDR}"
          return 0
        fi
        ;;
    esac
  else
    log fatal "${FUNCNAME}: No default route. (err: $res)"
  fi

}
#}}}

# Check dns resolution {{{
#
# Test DNS resolution
# $1 : FQDN to test on
#
# Echo : host command return code

function check_dns_resolution () {

  local _DNS_NAME="${1}"

  log info "Testing Dns resolution ( host ${_DNS_NAME} > /dev/null )"
  host ${_DNS_NAME} > /dev/null
  res=$?

  [ $res -ne 0 ] && log debug "${FUNCNAME}: Dns resolution not available. (err: $res)"

  return $res
}
#}}}

# Check remote service {{{
#
# Check if remote service is available
#
# $1 : Remote IP or FQDN
# $2 : Protocol <UDP|TCP>
# $3 : Remote port
#
# Echo : nc command return code
function check_remote_service () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "3" "$#"

  local _HOST="${1}"
  local _PROTO="${2}"
  local _PORT="${3}"

  # Netcat default is on TCP proto we have to set "-u"
  # option if UDP needed.
  # Lowercase ${_PROTO} to avoid case sensitive

  local _PROTO_ARG=""
  [ "${_PROTO,,}" = "udp" ] && _PROTO_ARG="-u -v"

  nc -z -w 5 ${_PROTO_ARG} ${_HOST} ${_PORT} > /dev/null 2>&1
  res=$?
  if [ $res -eq 0 ]
  then
    log info "Remote port ${_PORT} proto ${_PROTO} on ${_HOST} is available"
    return $res
  else
    log warn "Remote port ${_PORT} proto ${_PROTO} on ${_HOST} is not available"
    return $res
  fi
}
#}}}

# Check if remote is reachable {{{
#
# $1 : Remote address
# $2 : Type < FQDN | IP >
#
# Echo : return code
function is_remote_reachable () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "1" "$#"

  local _REMOTE="${1}"

  ping -c 1 ${_REMOTE} > /dev/null 2>&1
  res=$?

  case $res in
    0)
      log info "Can ping remote ${_REMOTE}"
      ;;
    1)
      log debug "${FUNCNAME}: Can't ping remote ${_REMOTE}, server down ?. (err: $res)"
      ;;
    *)
      log debug "${FUNCNAME}: Got ping error. (err: $res)"
      ;;
  esac

  return $res
}
#}}}

# Check if network is reachable {{{
#
# Echo : return code
function is_network_reachable () {

  ip ro get 1.2.3.4 > /dev/null 2>&1
  res=$?

  case $res in
    0)
      log info "Network is reachable"
      ;;
    *)
      log Debug "${FUNCNAME}: Network is unreachable. (err: $res)"
      ;;
  esac

  return $res

}
#}}}
#}}}

# Time randomization {{{
#
# $1: Min sleep time in sec
# $2: Max sleep time in sec
#
# Return : sleep time ins
function random_sleep_time () {
  local _MIN_SLEEP_TIME=${1}
  local _MAX_SLEEP_TIME=${2}
  local _SLEEP_TIME

  if [ -n "${RANDOM}" ]
  then
    _SLEEP_TIME=$((${RANDOM} * $((${_MAX_SLEEP_TIME} - ${_MIN_SLEEP_TIME})) / 32767 + ${_MIN_SLEEP_TIME}))
  else
    _SLEEP_TIME=0
    while [ "${_SLEEP_TIME}" -lt "${_MIN_SLEEP_TIME}" -o "${_SLEEP_TIME}" -gt "${_MAX_SLEEP_TIME}" ] ; do
       _SLEEP_TIME=`head -1 /dev/urandom | cksum | awk '{print $2}'`
    done
  fi

  echo ${_SLEEP_TIME}
}
#}}}

# Get package satus {{{
#
# $1 : Package name
#
# Echo package status <available|installed|upgradable>
function get_package_status () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "1" "$#"

  local _PACKAGE="${1}"
  local _STATE

  read _name1 _installed _name2 _candidate <<< "$(echo `LC_ALL=POSIX apt-cache policy ${_PACKAGE} | grep -E "^\s+(Installed|Candidate):"`)"

  if [ "$_installed" == "(none)" ]
  then
    _STATE="not installed"
  elif [ "$_installed" != "$_candidate" ]
  then
    _STATE="upgradable"
  else
    _STATE="uptodate"
  fi

  echo "${_STATE}"

}
#}}}

# Get installed package version {{{
#
# Return installed package version
#
# $1 : package name
#
# Echo : package name and version

function get_version () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "1" "$#"

  local _PKG_NAME=${1}

  local _VERSION=`dpkg -l ${_PKG_NAME} | grep -E "^i[[:alpha:]]" | awk '{print $3}'`
  echo "${_PKG_NAME} version : ${_VERSION}"
}
#}}}

# Get installed package version and status {{{
#
# Return installed package version
#
# $1 : package name
# $2 : option <version|status>
#
# Echo : version or array of state info

function get_package_info () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "2" "$#"

  local _PKG_NAME=${1}
  local _OPTION="${2}"
  local _STATE _VERSION _DESIRED _STATUS _ERROR
  read _STATE _VERSION  < <(dpkg -l ${_PKG_NAME} | tail -1 | awk '{print $1, $3}')

  if [ ${_STATE}  ]
  then
    declare -A _DESIRED_INFO=([i]=Install [r]=Remove [p]=Purge [h]=Hold)
    declare -A _STATUS_INFO=([n]="Not installed" [i]=Installed [c]="Configuration files" \
                             [u]=Unpacked [f]="Failed to remove configuration files" \
                             [h]="Half installed" [W]="Awaiting trigger" [t]="Trigger pending")
  fi

  case ${_OPTION} in
    version)
      echo "${_PKG_NAME} version : ${_VERSION}"
      ;;
    status)
      _DESIRED=${_STATE:0:1}
      _STATUS=${_STATE:1:1}
      _ERROR=${_STATE:2:1}

      declare _PKG_INFO=("${_PKG_NAME}" "Desired: ${_DESIRED_INFO[${_DESIRED}]}" \
                          "Status:${_STATUS_INFO[${_STATUS}]}" "Error:${_ERROR}")

      # output array as string using 'declare's representation (minus header)
      declare -p _PKG_INFO | sed -e 's/^declare -a [^=]*=//'
      ;;
  esac

}
#}}}

# Manage remote files or scripts to donwload {{{
# Init dirs{{{
function init_dirs () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "1" "$#"

  local _LOCAL_SCRIPT_DIR="${1}"

  if [ ! -d ${_LOCAL_SCRIPT_DIR} ]
  then
    log info "${_LOCAL_SCRIPT_DIR} not present, creating"
    run_and_log mkdir ${LOCAL_SCRIPT_DIR}
    res=$?
    [ $res -ne 0 ] && log_and_mail_error "${FUNCNAME}: can't mkdir ${_LOCAL_SCRIPT_DIR} (err: $res)"
  fi

  return 0
}
#}}}

# Update state file {{{
function init_list_file_statefile () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "1" "$#"

  local _STATEFILE="${1}"

  # Check if state file exist
  # else create.

  if [ ! -f ${_STATEFILE} -o ! -s ${_STATEFILE} ]
  then
      log info "Initialising ${_STATEFILE}"
      cat << EOF > ${_STATEFILE}
# This file liste files download or executed to upgrade
# postes in production without any error.
#
# File Name - Download or execution date

EOF
    return 0
  elif [ -f ${_STATEFILE} -a -s ${_STATEFILE} ]
  then
    log info "${_STATEFILE} exists and is not empty"
    return 0
  else
    return 1
  fi

}
#}}}

# Parse wget exit status {{{
function wget_exit_status () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "3" "$#"

  local _EXIT_STATUS="${1}"
  local _FUNCNAME="${2}"
  local _URL="${3}"

  local _ERR_MSG

  case ${_EXIT_STATUS} in
    1)
      _ERR_MSG="Generic errror code."
      ;;
    2)
      _ERR_MSG="commmand line option error."
      ;;
    3)
      _ERR_MSG="file I/O error."
      ;;
    4)
      _ERR_MSG="network failure."
      ;;
    5)
      _ERR_MSG="SSL verification failure."
      ;;
    6)
      _ERR_MSG="username/password auth failure"
      ;;
    7)
      _ERR_MSG="protocol errors."
      ;;
    8)
      _ERR_MSG="remote server issued an error response"
      ;;
  esac

  log fatal "${_FUNCNAME}: Failed to download ${_URL} - Wget: ${_ERR_MSG}"
}

#}}}
# Get remote files list{{{
function list_remote_files () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "3" "$#"

  local _USER="${1}"
  local _PASS="${2}"
  local _URL="${3}"

  local _TMP_FILE="$( mktemp )"

  run_and_log wget --no-check-certificate --http-user "${_USER}" --http-password "${_PASS}" -q -O "${_TMP_FILE}" "${_URL}"
  res=$?

  [ $res -ne 0 ] && wget_exit_status $res ${FUNCNAME} ${_URL}

  # load list in array
  while read line
  do
    _REMOTE_LIST+=($line)
  done < ${_TMP_FILE}

  # Remove tmp file.
  rm -f ${_TMP_FILE}

  # output array as string using 'declare's representation (minus header
  declare -p _REMOTE_LIST | sed -e 's/^declare -a [^=]*=//'
}
#}}}
# Get list of remote files already downloaded {{{
function list_remote_downloaded_files () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "1" "$#"

  local _UPDATESFILE="${1}"

  # Get list
  while read line
  do
    [ -z "$line" ] && continue
    [ $( echo $line | egrep -c "^#" ) -ge 1 ] && continue
    [ $( echo $line | egrep -c "^\s$" ) -ge 1 ] && continue
    [ $( echo $line | egrep -c "^[a-zA-Z0-9]+" ) -eq 0 ] && continue

    # Push in array without the date
    read script date < <(echo $line)
    _LOCAL_SCRIPT_LIST+=( $script )
  done < ${_UPDATESFILE}

  # output array as string using 'declare's representation (minus header
  declare -p _LOCAL_SCRIPT_LIST | sed -e 's/^declare -a [^=]*=//'
}
#}}}

# Compare array and retrieve uniq entries {{{
function compare_array_uniq () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "2" "$#"

  local _REF_ARRAY_1="$1[@]" # array
  local _ARRAY_1=("${!_REF_ARRAY_1}") # array

  local _REF_ARRAY_2="$2[@]" # array
  local _ARRAY_2=("${!_REF_ARRAY_2}") # array

  local _ARRAY_OF_UNIQ=(`echo ${_ARRAY_1[@]} ${_ARRAY_2[@]} | tr ' ' '\n' | sort | uniq -u`)

  declare -p _ARRAY_OF_UNIQ | sed -e 's/^declare -a [^=]*=//'

}
#}}}
# Compare array and retrieve duplicate entries {{{
function compare_array_duplicate () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "2" "$#"

  local _REF_ARRAY_1="$1[@]" # array
  local _ARRAY_1=("${!_REF_ARRAY_1}") # array

  local _REF_ARRAY_2="$2[@]" # array
  local _ARRAY_2=("${!_REF_ARRAY_2}") # array

  local _ARRAY_OF_DUPLICATE=(`echo ${_ARRAY_1[@]} ${_ARRAY_2[@]} | tr ' ' '\n' | sort | uniq -d`)

  declare -p _ARRAY_OF_DUPLICATE | sed -e 's/^declare -a [^=]*=//'

}
#}}}
# Compare file's list available from remote with the one allready downloaded {{{
function compare_remote_and_local_file_list () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "2" "$#"

  local _ARRAY_1="$1[@]" # array
  local _REMOTE_LIST=("${!_ARRAY_1}") # array

  local _ARRAY_2="$2[@]" # array
  local _LOCAL_LIST=("${!_ARRAY_2}") # array

  for remote_name in "${_REMOTE_LIST[@]}"
  do
    for local_name in "${_LOCAL_LIST[@]}"
    do
      if [ "$remote_name" == "$local_name" ]
      then
        #log info "Script $remote_name allready applied"
        continue 2
      fi
    done
    _DOWNLOAD_LIST+=($remote_name)
  done

  # output array as string using 'declare's representation (minus header
  declare -p _DOWNLOAD_LIST | sed -e 's/^declare -a [^=]*=//'
}
#}}}

# Donwload remote files from list {{{
function download_remote_files () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "5" "$#"

  local _USER="${1}"
  local _PASS="${2}"
  local _URL="${3}"
  local _LOCAL_DIR="${4}"
  local _ARRAY="$5[@]"
  local _DOWNLOAD_LIST=("${!_ARRAY}") # array

  local _FILE_PATH _FILE_NAME

  for _FILE_NAME in "${_DOWNLOAD_LIST[@]}"
  do
    _FILE_PATH="${_LOCAL_DIR}/${_FILE_NAME}"
    run_and_log wget --no-check-certificate --http-user "${_USER}" \
                      --http-password "${_PASS}" -q -O ${_FILE_PATH} ${_URL}/${_FILE_NAME}

    res=$?
    [ $res -ne 0 ] && wget_exit_status $res ${FUNCNAME} ${_URL}/${_FILE_NAME}

  done

}
#}}}

# Update state file {{{
function update_list_file_statefile () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "3" "$#"

  local _DATE="${1}"
  local _STATEFILE="${2}"
  local _ARRAY="${3}"
  local _APPLIED_FILES_LIST=("${!_ARRAY}")

  local _FILE

  log info "Update ${_STATEFILE} with new files donwloaded or executed without error"
  for _FILE in "${_APPLIED_FILES_LIST[@]}"
  do
    log info  "Run : echo -e \"${_FILE}\\t\\t${DATE}\" >> ${_STATEFILE}"
    echo -e "${_FILE}\t\t${DATE}" >> ${_STATEFILE}
    res=$?
    [ $res -ne 0 ] && log_and_mail_error "${FUNCMANE}: Can't update ${_STATEFILE} (err: $res)"

    log info "Line: \"${_FILE}  ${_DATE}\" added to ${_STATEFILE}"
  done

  return 0
}
#}}}
#}}}

# Update alternatives {{{
#
# $1 : Applcation name
# $2 : Application PATH
# $3 : Alternative name
#
function update_alternatives () {

  local _APP_NAME=${1}
  local _APP_PATH=${2}
  local _ALT_NAME=${3}

  if [ -f ${_APP_PATH} ]
  then
    log info "${_APP_NAME} installed, check if it's default ${_ALT_NAME}"
    # Test if allready set
    current_default_alternative=`update-alternatives --display ${_ALT_NAME} | grep "link currently points to" | awk '  {print $NF}'`
    if [ "${current_default_alternative}" == "${_APP_PATH}" ]
    then
      log info "${_APP_NAME} allready as default ${_ALT_NAME}"
    else
      log info "Setting ${_APP_NAME} as default ${_ALT_NAME}"
      run_and_log 'update-alternatives --quiet --set ${_ALT_NAME} ${_APP_PATH}'
      res=$?
      [ $res -ne 0 ] && log warn "${FUNCNAME}: Error while setting ${_APP_NAME} as default ${_ALT_NAME}. (err: $res)"
    fi
  else
    log info "${_APP_NAME} not installed, can't set it as default ${_ALT_NAME}"
  fi
}
#}}}

# Log function exit status on error {{{
function exit_status () {

  # Check args passed
  check_args "${0}" "${FUNCNAME}" "4" "$#"

  local _LEVEL=${1}
  local _FUNCTION=${2}
  local _STATUS=${3}
  local _MSG="${4}"

  [ -z "${_MSG}" ] && MSG="exit with"

  log ${_LEVEL} "${_FUNCTION}: ${_MSG} (status ${_STATUS})"
}
#}}}

# vim:expandtab:shiftwidth=2:softtabstop=2:tabstop=2:noexpandtab:foldmethod=marker:foldlevel=0
